package edu.unika.aifb.kaon.apionrdf.exporter;

import java.util.Set;
import java.util.Iterator;
import java.io.IOException;
import java.io.Writer;

import edu.unika.aifb.rdf.api.syntax.*;
import edu.unika.aifb.rdf.api.util.*;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * An exporter for wiritng KAON ontologies.
 */
public class KAONExporter extends AbstractExporter {
    /** The RDF writer. */
    protected RDFWriter m_rdfWriter;

    /**
     * Creates an instance of this class.
     */
    public KAONExporter() {
        m_rdfWriter=new RDFWriter();
    }
    /**
     * Serializes given RDF model.
     *
     * @param oimodel                   the OI-model to serialize
     * @param physicalURI               the physical URI of the exported file
     * @param writer                    the writer where serialization is performed
     * @param encoding                  tne encoding
     * @throws IOException              thrown if there is an error
     * @throws KAONException            thrown if there is an error accessing the OI-model
     * @throws InterruptedException     thrown if the process is interrupted
     */
    public void export(OIModel oimodel,String physicalURI,Writer writer,String encoding) throws IOException,KAONException,InterruptedException {
        try {
            m_rdfWriter.prepareNamespaceCollection();
            m_rdfWriter.collectNamespace(RDFConstants.RDFS_CLASS);
            m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,0,3);
            Set concepts=oimodel.getConcepts();
            collectNamespaces(concepts);
            checkInterrupted();
            m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,1,3);
            Set properties=oimodel.getProperties();
            collectNamespaces(properties);
            checkInterrupted();
            m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,2,3);
            Set instances=oimodel.getInstances();
            collectNamespaces(instances);
            checkInterrupted();
            m_progressListener.processorProgress(PHASE_LOAD_OBJECTS,3,3);
            m_rdfWriter.startSerialization(writer,physicalURI,oimodel.getLogicalURI(),encoding);
            writeInclusions(oimodel);
            writeModelAttributes(oimodel);
            m_rdfWriter.startRDFContents();
            processElements(concepts,oimodel,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
                protected void processObject(Entity entity) throws KAONException,IOException {
                    writeConcept((Concept)entity);
                }
            },PHASE_WRITE_CONCEPTS);
            concepts=null; // to free some memory
            processElements(properties,oimodel,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
                protected void processObject(Entity entity) throws KAONException,IOException {
                    writeProperty((Property)entity);
                }
            },PHASE_WRITE_PROPERTIES);
            properties=null; // to free some memory
            processElements(instances,oimodel,OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES | OIModel.LOAD_LEXICON,new SingleObjectProcessor() {
                protected void processObject(Entity entity) throws KAONException,IOException {
                    writeInstance((Instance)entity);
                }
            },PHASE_WRITE_INSTANCES);
            instances=null; // to free some memory
            m_rdfWriter.finishRDFContents();
        }
        finally {
            m_rdfWriter.cleanUp();
        }
    }
    /**
     * Writes the inclusion statements.
     *
     * @param oimodel                   the OI-model
     * @throws IOException              thrown if there is an I/O error
     * @throws KAONException            thrown if there is an error accessing the OI-model
     */
    protected void writeInclusions(OIModel oimodel) throws IOException,KAONException {
        Set includedOIModels=oimodel.getIncludedOIModels();
        if (!includedOIModels.isEmpty()) {
            m_rdfWriter.getOut().println();
            Iterator iterator=includedOIModels.iterator();
            while (iterator.hasNext()) {
                OIModel includedOIModel=(OIModel)iterator.next();
                m_rdfWriter.writeInclusion(includedOIModel.getLogicalURI(),includedOIModel.getPhysicalURI());
            }
        }
    }
    /**
     * Writes the OI-model attributes.
     *
     * @param oimodel                   the OI-model
     * @throws KAONException            thrown if there is an error accessing the OI-model
     */
    protected void writeModelAttributes(OIModel oimodel) throws KAONException {
        m_rdfWriter.writeModelAttributes(oimodel.getAttributes());
    }
    /**
     * Processes the names of entities from given set.
     *
     * @param set                       the set
     * @throws KAONException            thrown if there is an error
     */
    protected void collectNamespaces(Set set) throws KAONException {
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            Entity entity=(Entity)iterator.next();
            m_rdfWriter.collectNamespace(entity.getURI());
        }
    }
    /**
     * Writes the concept.
     *
     * @param concept                   the concept to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeConcept(Concept concept) throws KAONException,IOException {
        if (shouldSerializeEntity(concept) && (!shouldSerializeOnlyLocalElements() || concept.isDeclaredLocally())) {
            m_rdfWriter.writeStatement(concept.getURI(),RDFConstants.RDF_TYPE,RDFConstants.RDFS_CLASS,null,null,false);
            writeLexicalEntries(concept);
        }
        Iterator iterator=concept.getSuperConcepts().iterator();
        while (iterator.hasNext()) {
            Concept superConcept=(Concept)iterator.next();
            if (shouldSerializeEntity(superConcept) && (!shouldSerializeOnlyLocalElements() || concept.isSuperSubConceptDeclaredLocally(superConcept)))
                m_rdfWriter.writeStatement(concept.getURI(),RDFConstants.RDFS_SUBCLASSOF,superConcept.getURI(),null,null,false);
        }
    }
    /**
     * Writes the property.
     *
     * @param property                  the property to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeProperty(Property property) throws KAONException,IOException {
        if (shouldSerializeEntity(property) && (!shouldSerializeOnlyLocalElements() || property.isDeclaredLocally())) {
            m_rdfWriter.writeStatement(property.getURI(),RDFConstants.RDF_TYPE,RDFConstants.RDF_PROPERTY,null,null,false);
            writeLexicalEntries(property);
            writeKAONSpecificPropertyInformation(property);
            if (property.isAttribute())
                m_rdfWriter.writeStatement(property.getURI(),RDFConstants.RDFS_RANGE,KAONVocabularyAdaptor.INSTANCE.getLiteral(),null,null,false);
        }
        Iterator iterator=property.getSuperProperties().iterator();
        while (iterator.hasNext()) {
            Property superProperty=(Property)iterator.next();
            if (shouldSerializeEntity(superProperty) && (!shouldSerializeOnlyLocalElements() || property.isSuperSubPropertyDeclaredLocally(superProperty)))
                m_rdfWriter.writeStatement(property.getURI(),RDFConstants.RDFS_SUBPROPERTYOF,superProperty.getURI(),null,null,false);
        }
        iterator=property.getDomainConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (shouldSerializeEntity(concept) && (!shouldSerializeOnlyLocalElements() || property.isDomainConceptDeclaredLocally(concept)))
                m_rdfWriter.writeStatement(property.getURI(),RDFConstants.RDFS_DOMAIN,concept.getURI(),null,null,false);
        }
        iterator=property.getRangeConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (shouldSerializeEntity(concept) && (!shouldSerializeOnlyLocalElements() || property.isRangeConceptDeclaredLocally(concept)))
                m_rdfWriter.writeStatement(property.getURI(),RDFConstants.RDFS_RANGE,concept.getURI(),null,null,false);
        }
    }
    /**
     * Writes the KAON-specific property information.
     *
     * @param property                  the property
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeKAONSpecificPropertyInformation(Property property) throws KAONException,IOException {
        if (property.isInversePropertyDeclaredLocally()) {
            Property inverseProperty=property.getInverseProperty();
             if (inverseProperty!=null && !shouldSerializeEntity(inverseProperty))
                m_rdfWriter.writeStatement(property.getURI(),KAONVocabularyAdaptor.INSTANCE.getInverse(),inverseProperty.getURI(),null,null,false);
        }
        if (property.isSymmetric())
            m_rdfWriter.writeStatement(property.getURI(),KAONVocabularyAdaptor.INSTANCE.getSymmetric(),"true",null,null,true);
        if (property.isTransitive())
            m_rdfWriter.writeStatement(property.getURI(),KAONVocabularyAdaptor.INSTANCE.getTransitive(),"true",null,null,true);
    }
    /**
     * Writes the instance.
     *
     * @param instance                  the instance to write
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeInstance(Instance instance) throws KAONException,IOException {
        if (shouldSerializeEntity(instance) && !instance.getSpanningConcept().isInOIModel() && !instance.getSpanningProperty().isInOIModel() && (!shouldSerializeOnlyLocalElements() || instance.isDeclaredLocally()))
            writeLexicalEntries(instance);
        Iterator iterator=instance.getParentConcepts().iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (!KAONVocabularyAdaptor.INSTANCE.getRoot().equals(concept.getURI()) && shouldSerializeEntity(concept) && (!shouldSerializeOnlyLocalElements() || instance.isConceptInstanceDeclaredLocally(concept)))
                m_rdfWriter.writeStatement(instance.getURI(),RDFConstants.RDF_TYPE,concept.getURI(),null,null,false);
        }
        iterator=instance.getFromPropertyInstances().iterator();
        while (iterator.hasNext()) {
            PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
            if (shouldSerializePropertyInstance(propertyInstance) && (!shouldSerializeOnlyLocalElements() || propertyInstance.isDeclaredLocally())) {
                Object targetValue=propertyInstance.getTargetValue();
                if (targetValue instanceof Instance)
                    m_rdfWriter.writeStatement(instance.getURI(),propertyInstance.getProperty().getURI(),((Instance)targetValue).getURI(),null,null,false);
                else
                    m_rdfWriter.writeStatement(instance.getURI(),propertyInstance.getProperty().getURI(),(String)targetValue,null,null,true);
            }
        }
    }
    /**
     * Processes the lexical entries attached to the entity.
     *
     * @param entity                    the entity
     * @throws KAONException            thrown if there is an error
     * @throws IOException              thrown if there is an error
     */
    protected void writeLexicalEntries(Entity entity) throws KAONException,IOException {
    }
    /**
     * Returns <code>true</code> if at serialization only current model should be serialized.
     *
     * @return                          <code>true</code> if only elements declared in local model should be serialzied
     */
    protected boolean shouldSerializeOnlyLocalElements() {
        return true;
    }
    /**
     * Return <code>true</code> if only given entity should be serialized.
     *
     * @param entity                    the entity being serialized
     * @return                          <code>true</code> if the entity should be serialized
     * @throws KAONException            thrown if there is an error
     */
    protected boolean shouldSerializeEntity(Entity entity) throws KAONException {
        return true;
    }
    /**
     * Returns <code>true</code> if property instance should be serialized.
     *
     * @param propertyInstance          the property instance
     * @return                          <code>true</code> if the property instance should be serialized
     * @throws KAONException            thrown if there is an error
     */
    protected boolean shouldSerializePropertyInstance(PropertyInstance propertyInstance) throws KAONException {
        return true;
    }

    /**
     * The processor for single objects.
     */
    protected abstract class SingleObjectProcessor implements ObjectProcessor {
        public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
            try {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Entity entity=(Entity)iterator.next();
                    processObject(entity);
                    checkInterrupted();
                }
                checkInterrupted();
            }
            catch (IOException e) {
                throw new KAONException("I/O error",e);
            }
        }
        protected abstract void processObject(Entity entity) throws KAONException,IOException;
    }
}
