package edu.unika.aifb.kaon.apionrdf;

import java.util.List;

import edu.unika.aifb.kaon.datalog.program.*;

/**
 * This static class contains a Datalog program that provides the standard axiomatization of the KAON API over RDF.
 * The consists of the following rules:
 * <p><code>
 * ConceptHierarchy(X,Y) :- Triple(Y,rdfs:subClassOf,X)<br>
 * <br>
 * ConceptHierarchy_t(X,Y) :- Triple(Y,rdfs:subClassOf,X)<br>
 * ConceptHierarchy_t(X,Y) :- Triple(Z,rdfs:subClassOf,X),ConceptHierarchy_t(Z,Y)<br>
 * <br>
 * PropertyHierarchy(X,Y) :- Triple(Y,rdf:subPropertyOf,X)<br>
 * <br>
 * PropertyHierarchy_t(X,Y) :- Triple(Y,rdfs:subPropertyOf,X)<br>
 * PropertyHierarchy_t(X,Y) :- Triple(Z,rdfs:subPropertyOf,X),PropertyHierarchy_t(Z,Y)<br>
 * <br>
 * ConceptInstance(C,I) :- Triple(I,rdf:type,C)<br>
 * <br>
 * ConceptInstance_a(C,I) :- Triple(I,rdf:type,C)<br>
 * ConceptInstance_a(C,I) :- ConceptHierarchy_t(C,SC),Triple(I,rdf:type,SC)<br>
 * <br>
 * AttributeInstance(P,S,V) :- Triple(S,P,V),IsLiteral(V),IsVisible(P)<br>
 * <br>
 * AttributeInstance_a(P,S,V) :- AttributeInstance(P,S,V,null)<br>
 * AttributeInstance_a(P,S,V) :- PropertyHierarchy_t(P,SP),AttributeInstance(SP,S,V,null)<br>
 * <br>
 * RelationInstance(P,S,T) :- Triple(S,P,T),IsResource(T),IsVisible(P)<br>
 * <br>
 * RelationInstance_a(P,S,T) :- RelationInstance(P,S,T)<br>
 * RelationInstance_a(P,S,T) :- PropertyHierarchy_t(P,SP),RelationInstance_a(SP,S,T)<br>
 * RelationInstance_a(P,S,T) :- Triple(P,kaon:inverse,IP),RelationInstance_a(IP,T,S)<br>
 * RelationInstance_a(P,S,T) :- Triple(P,kaon:symmetric,true),RelationInstance(P,T,S)<br>
 * RelationInstance_a(P,S,T) :- Triple(P,kaon:transitive,true),RelationInstance_a(P,S,M),RelationInstance_a(P,M,T)<br>
 * <br>
 * PropertyInstance_a(P,S,NULL,V) :- AttributeInstance_a(P,S,V)<br>
 * PropertyInstance_a(P,S,T,NULL) :- RelationInstance_a(P,S,T)
 * </code></p>
 */
public class StandardAxiomatization {
    /**
     * Returns the axiomatization for given OI-model.
     *
     * @param oimodel                               the OI-model
     * @param predicateFactory                      the predicate factory to use
     * @param rules                                 the list receiving the rules
     */
    public static void getAxiomatization(OIModelImpl oimodel,PredicateFactory predicateFactory,List rules) {
        // EDB predicates
        Predicate triple=predicateFactory.getPredicate("Triple",3);
        Predicate isResource=predicateFactory.getPredicate("IsResource",1);
        Predicate isLiteral=predicateFactory.getPredicate("IsLiteral",1);
        Predicate isVisible=predicateFactory.getPredicate("IsVisible",1);
        // IDB predicates that are often EDB predicates
        Predicate conceptHierarchy=predicateFactory.getPredicate("ConceptHierarchy",2);
        Predicate conceptInstance=predicateFactory.getPredicate("ConceptInstance",2);
        Predicate propertyHierarchy=predicateFactory.getPredicate("PropertyHierarchy",2);
        Predicate propertyDomain=predicateFactory.getPredicate("PropertyDomain",4);
        Predicate propertyRange=predicateFactory.getPredicate("PropertyRange",2);
        Predicate attributeInstance=predicateFactory.getPredicate("AttributeInstance",3);
        Predicate relationInstance=predicateFactory.getPredicate("RelationInstance",3);
        // IDB predicates
        Predicate conceptHierarchy_t=predicateFactory.getPredicate("ConceptHierarchy_t",2);
        Predicate propertyHierarchy_t=predicateFactory.getPredicate("PropertyHierarchy_t",2);
        Predicate conceptInstance_a=predicateFactory.getPredicate("ConceptInstance_a",2);
        Predicate attributeInstance_a=predicateFactory.getPredicate("AttributeInstance_a",3);
        Predicate relationInstance_a=predicateFactory.getPredicate("RelationInstance_a",3);
        Predicate propertyInstance_a=predicateFactory.getPredicate("PropertyInstance_a",4);
        // utility variables
        Variable X=new Variable("X");
        Variable Y=new Variable("Y");
        Variable Z=new Variable("Z");
        Variable C=new Variable("C");
        Variable SC=new Variable("SC");
        Variable I=new Variable("I");
        Variable P=new Variable("P");
        Variable SP=new Variable("SP");
        Variable IP=new Variable("IP");
        Variable S=new Variable("S");
        Variable T=new Variable("T");
        Variable V=new Variable("V");
        Variable M=new Variable("M");
        // the program
        rules.add(new Rule("ch",
            new Literal(conceptHierarchy,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(triple,true,new Term[] { Y,new Constant(oimodel.m_resourceSubClassOf),X }),
            }
        ));

        rules.add(new Rule("ch1",
            new Literal(conceptHierarchy_t,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(conceptHierarchy,true,new Term[] { X,Y }),
            }
        ));
        rules.add(new Rule("ch2",
            new Literal(conceptHierarchy_t,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(conceptHierarchy,true,new Term[] { X,Z }),
                new Literal(conceptHierarchy_t,true,new Term[] { Z,Y }),
            }
        ));

        rules.add(new Rule("pd",
            new Literal(propertyDomain,true,new Term[] { X,Y,new Constant(oimodel.m_literalZero),new Constant(oimodel.m_literalInfinity) }),
            new Literal[] {
                new Literal(triple,true,new Term[] { X,new Constant(oimodel.m_resourceDomain),Y }),
            }
        ));

        rules.add(new Rule("pr",
            new Literal(propertyRange,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(triple,true,new Term[] { X,new Constant(oimodel.m_resourceRange),Y }),
            }
        ));

        rules.add(new Rule("ph",
            new Literal(propertyHierarchy,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(triple,true,new Term[] { Y,new Constant(oimodel.m_resourceSubPropertyOf),X }),
            }
        ));

        rules.add(new Rule("ph1",
            new Literal(propertyHierarchy_t,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(propertyHierarchy,true,new Term[] { X,Y }),
            }
        ));
        rules.add(new Rule("ph2",
            new Literal(propertyHierarchy_t,true,new Term[] { X,Y }),
            new Literal[] {
                new Literal(propertyHierarchy,true,new Term[] { X,Z }),
                new Literal(propertyHierarchy_t,true,new Term[] { Z,Y }),
            }
        ));

        rules.add(new Rule("ci",
            new Literal(conceptInstance,true,new Term[] { C,I }),
            new Literal[] {
                new Literal(triple,true,new Term[] { I,new Constant(oimodel.m_resourceInstanceOf),C }),
            }
        ));

        rules.add(new Rule("ci1",
            new Literal(conceptInstance_a,true,new Term[] { C,I }),
            new Literal[] {
                new Literal(conceptInstance,true,new Term[] { C,I }),
            }
        ));
        rules.add(new Rule("ci2",
            new Literal(conceptInstance_a,true,new Term[] { C,I }),
            new Literal[] {
                new Literal(conceptHierarchy_t,true,new Term[] { C,SC }),
                new Literal(conceptInstance,true,new Term[] { SC,I }),
            }
        ));

        rules.add(new Rule("ai",
            new Literal(attributeInstance,true,new Term[] { P,S,V }),
            new Literal[] {
                new Literal(triple,true,new Term[] { S,P,V }),
                new Literal(isLiteral,true,new Term[] { V }),
                new Literal(isVisible,true,new Term[] { P }),
            }
        ));

        rules.add(new Rule("ai1",
            new Literal(attributeInstance_a,true,new Term[] { P,S,V }),
            new Literal[] {
                new Literal(attributeInstance,true,new Term[] { P,S,V }),
            }
        ));
        rules.add(new Rule("ai2",
            new Literal(attributeInstance_a,true,new Term[] { P,S,V }),
            new Literal[] {
                new Literal(propertyHierarchy_t,true,new Term[] { P,SP }),
                new Literal(attributeInstance,true,new Term[] { SP,S,V }),
            }
        ));

        rules.add(new Rule("ri",
            new Literal(relationInstance,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(triple,true,new Term[] { S,P,T }),
                new Literal(isResource,true,new Term[] { T }),
                new Literal(isVisible,true,new Term[] { P }),
            }
        ));

        rules.add(new Rule("ri1",
            new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(relationInstance,true,new Term[] { P,S,T }),
            }
        ));
        rules.add(new Rule("ri2",
            new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(propertyHierarchy_t,true,new Term[] { P,SP }),
                new Literal(relationInstance_a,true,new Term[] { SP,S,T }),
            }
        ));
        rules.add(new Rule("ri3",
            new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(triple,true,new Term[] { P,new Constant(oimodel.m_resourceInverse),IP }),
                new Literal(relationInstance_a,true,new Term[] { IP,T,S }),
            }
        ));
        rules.add(new Rule("ri4",
            new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(triple,true,new Term[] { P,new Constant(oimodel.m_resourceSymmetric),new Constant(oimodel.m_literalTrue) }),
                new Literal(relationInstance,true,new Term[] { P,T,S }),
            }
        ));
        rules.add(new Rule("ri5",
            new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            new Literal[] {
                new Literal(triple,true,new Term[] { P,new Constant(oimodel.m_resourceTransitive),new Constant(oimodel.m_literalTrue) }),
                new Literal(relationInstance_a,true,new Term[] { P,S,M }),
                new Literal(relationInstance_a,true,new Term[] { P,M,T }),
            }
        ));

        rules.add(new Rule("pi1",
            new Literal(propertyInstance_a,true,new Term[] { P,S,new Constant(),V }),
            new Literal[] {
                new Literal(attributeInstance_a,true,new Term[] { P,S,V }),
            }
        ));
        rules.add(new Rule("pi2",
            new Literal(propertyInstance_a,true,new Term[] { P,S,T,new Constant() }),
            new Literal[] {
                new Literal(relationInstance_a,true,new Term[] { P,S,T }),
            }
        ));
    }
}
