package edu.unika.aifb.kaon.apionrdf;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of an instance of a property.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class PropertyInstanceImpl implements PropertyInstance {
    /** Property that this object is instance of. */
    protected PropertyImpl m_property;
    /** Source instance. */
    protected InstanceImpl m_sourceInstance;
    /** Target value (may be <code>Instance</code> or <code>String</code>. */
    protected Object m_targetValue;

    /**
     * Creates an instance of this class.
     *
     * @param property                          property
     * @param sourceInstance                    source instance of this property instance
     * @param targetValue                       target value (<code>String</code> or an <code>Instnace</code>)
     */
    public PropertyInstanceImpl(PropertyImpl property,InstanceImpl sourceInstance,Object targetValue) {
        m_property=property;
        m_sourceInstance=sourceInstance;
        m_targetValue=targetValue;
    }
    /**
     * Returns the property.
     *
     * @return                                  property
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Returns the instance that this relation points from.
     *
     * @return                                  source instance
     */
    public Instance getSourceInstance() {
        return m_sourceInstance;
    }
    /**
     * Returns the value of this relation instnace (may be <code>String</code> or <code>Instance</code>).
     *
     * @return                                  value of this relaiton instance (<code>String</code> or <code>Instnace</code>)
     */
    public Object getTargetValue() {
        return m_targetValue;
    }
    /**
     * Returns the OI-model of this object.
     *
     * @return                                  the OI-model of this object
     */
    public OIModel getOIModel() {
        return m_property.getOIModel();
    }
    /**
     * Checks whether this property instance is in the OI-model.
     *
     * @return                                  <code>true</code> if this property instance is in the OI-model.
     */
    public boolean isInOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=m_property.getModel();
                Resource subject=m_sourceInstance.getResource();
                Resource predicate=m_property.getResource();
                RDFNode object;
                if (m_targetValue instanceof InstanceImpl)
                    object=((InstanceImpl)m_targetValue).getResource();
                else
                    object=model.getNodeFactory().createLiteral((String)m_targetValue);
                return model.contains(subject,predicate,object);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            return m_property.containsStatementLocally(getMembershipStatement());
        }
    }
    /**
     * Returns the source OI-model of this object.
     *
     * @return                                  the source OI-model of this object
     */
    public OIModel getSourceOIModel() throws KAONException {
        synchronized (getLock()) {
            return m_property.getOIModelForStatement(getMembershipStatement());
        }
    }
    /**
     * Returns the statement determining membership of this property instance in the OI-model.
     *
     * @return                                  the statement for this property instance
     */
    protected Statement getMembershipStatement() throws KAONException {
        try {
            Model model=m_property.getModel();
            Resource subject=m_sourceInstance.getResource();
            Resource predicate=m_property.getResource();
            RDFNode object;
            if (m_targetValue instanceof InstanceImpl)
                object=((InstanceImpl)m_targetValue).getResource();
            else
                object=model.getNodeFactory().createLiteral((String)m_targetValue);
            return model.getNodeFactory().createStatement(subject,predicate,object);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns the lock object for this entity.
     *
     * @return                              the lock object for the entity
     */
    protected Object getLock() {
        return m_property.m_oimodel.m_kaonConnection;
    }
}
