package edu.unika.aifb.kaon.apionrdf;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of the Property interface that relies on the RDF API as the model.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class PropertyImpl extends AbstractEntityImpl implements Property {
    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param resource                  resource
     */
    public PropertyImpl(OIModelImpl oimodel,Resource resource) {
        super(oimodel,resource);
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        return OIModel.LOAD_PROPERTY_ALL;
    }
    /**
     * Tests whether this concept is part of its associated OI-model.
     *
     * @return                          <code>true</code> if concpet is a member in the OI-model
     */
    public boolean isInOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                return model.contains(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceProperty);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. Note that this may return <code>false</code> if this
     * entity was declared in an included OI-model.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                return model.thisContains(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceProperty);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model where this entity was declared originally.
     *
     * @return                          the OI-model there this entity was declated originally
     */
    public OIModel getSourceOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Statement statement=model.getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceProperty);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the inverse property of this property. If this property doesn't have an inverse property, then
     * <code>null</code> is returned.
     *
     * @return                          inverse property of this property (or <code>null</code> if there is no inverse property)
     */
    public Property getInverseProperty() throws KAONException {
        synchronized (getLock()) {
            try {
                Model result=getModel().find(m_resource,m_oimodel.m_resourceInverse,null);
                if (result.size()==0)
                    return null;
                else if (result.size()>1)
                    throw new KAONException("Property cannot have more than one inverse property.");
                else {
                    Statement statement=(Statement)result.iterator().next();
                    if (!(statement.object() instanceof Resource))
                        throw new KAONException("Property cannot have literal as inverse property.");
                    else
                        return m_oimodel.getProperty((Resource)statement.object());
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model in which the inverse property of this property has been declared.
     *
     * @return                          the OI-model in which the inverse property of this property has been declared
     */
    public OIModel getInversePropertyOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model result=getModel().find(m_resource,m_oimodel.m_resourceInverse,null);
                if (result.size()==0)
                    return null;
                else if (result.size()>1)
                    throw new KAONException("Property cannot have more than one inverse property.");
                else {
                    Statement statement=(Statement)result.iterator().next();
                    if (!(statement.object() instanceof Resource))
                        throw new KAONException("Property cannot have literal as inverse property.");
                    return getOIModelForStatement(statement);
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if the inverse property of this property has been specified within the OI-model as returned
     * by <code>getOIModel()</code>.
     *
     * @return                          <code>true</code> if the inverse property of this property has been specified whithin the OI-model as returned by <code>getOIModel()</code>
     */
    public boolean isInversePropertyDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            try {
                return getModel().thisContains(m_resource,m_oimodel.m_resourceInverse,null);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this property is symmetric.
     *
     * @return                          <code>true</code> if this property is symmetric
     */
    public boolean isSymmetric() throws KAONException {
        synchronized (getLock()) {
            try {
                Model result=getModel().find(m_resource,m_oimodel.m_resourceSymmetric,null);
                if (result.size()==0)
                    return false;
                else if (result.size()>1)
                    throw new KAONException("Property cannot have more than one symmetric property.");
                else {
                    Statement statement=(Statement)result.iterator().next();
                    if (statement.object() instanceof Resource)
                        throw new KAONException("Property must have a literal as the symmetric property.");
                    else
                        return m_oimodel.m_literalTrue.equals(statement.object());
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this property is transitive.
     *
     * @return                          <code>true</code> if this property is transitive
     */
    public boolean isTransitive() throws KAONException {
        synchronized (getLock()) {
            try {
                Model result=getModel().find(m_resource,m_oimodel.m_resourceTransitive,null);
                if (result.size()==0)
                    return false;
                else if (result.size()>1)
                    throw new KAONException("Property cannot have more than one transitive property.");
                else {
                    Statement statement=(Statement)result.iterator().next();
                    if (statement.object() instanceof Resource)
                        throw new KAONException("Property must have a literal as the transitive property.");
                    else
                        return m_oimodel.m_literalTrue.equals(statement.object());
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          array of concepts that consitute the domain of this property (may be empty)
     */
    public Set getDomainConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(m_resource,m_oimodel.m_resourceDomain,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Object object=statement.object();
                    if (object instanceof Resource)
                        set.add(m_oimodel.getConcept((Resource)object));
                    else
                        throw new KAONException("Domain cannot be a literal value.");
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          set of concepts that consitute the domain of this property (may be empty)
     */
    public Set getAllDomainConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getDomainConcepts());
            Iterator superProperties=getSuperProperties().iterator();
            while (superProperties.hasNext()) {
                Property property=(Property)superProperties.next();
                result.addAll(property.getAllDomainConcepts());
            }
            return result;
        }
    }
    /**
     * Checks whether supplied concept is in the domain of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the domain
     */
    public boolean isDomainConcept(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                return getModel().contains(m_resource,m_oimodel.m_resourceDomain,conceptID);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the domain to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the domain to this property
     */
    public OIModel getDomainConceptOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                Statement statement=getModel().getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceDomain,conceptID);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the domain to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the domain concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isDomainConceptDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                return getModel().thisContains(m_resource,m_oimodel.m_resourceDomain,conceptID);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the minimum cardinality for given domain concept.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          minimum cardinality of this property for given concept
     */
    public int getMinimumCardinality(Concept concept) throws KAONException {
        return getCardinality(concept,m_oimodel.m_resourceMinCardinality,0);
    }
    protected int getCardinality(Concept concept,Resource cardinalityResource,int defaultValue) throws KAONException {
        synchronized (getLock()) {
            try {
                Model result=getModel().find(m_resource,cardinalityResource,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Object object=statement.object();
                    if (object instanceof Resource)
                        throw new KAONException("The target of a cardinality property cannot be a resource.");
                    else {
                        String cardinalityLabel=((Literal)object).getLabel();
                        if (!cardinalityLabel.startsWith("["))
                            throw new KAONException("Invalid cardinality label '"+cardinalityLabel+"'.");
                        int terminatingBracket=cardinalityLabel.indexOf(']');
                        if (terminatingBracket==-1)
                            throw new KAONException("Invalid cardinality label '"+cardinalityLabel+"'.");
                        String conceptURI=cardinalityLabel.substring(1,terminatingBracket);
                        if (concept.getURI().equals(conceptURI))
                            try {
                                String numberString=cardinalityLabel.substring(terminatingBracket+1);
                                return Integer.parseInt(numberString);
                            }
                            catch (NumberFormatException e) {
                                throw new KAONException("Invalid cardinality label '"+cardinalityLabel+"'.");
                            }
                    }
                }
                return  defaultValue;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the maximum cardinality for given domain concept. In case of unlimited cardinality, Integer.MAX_VALUE is returned.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          maximum cardinality of this property for given concept
     */
    public int getMaximumCardinality(Concept concept) throws KAONException {
        return getCardinality(concept,m_oimodel.m_resourceMaxCardinality,Integer.MAX_VALUE);
    }
    /**
     * Return all concepts that consitute the range of this property.
     *
     * @return                          array of concepts that consitute the range of this property (may be empty)
     */
    public Set getRangeConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                if (isAttribute())
                    return set;
                Model result=getModel().find(m_resource,m_oimodel.m_resourceRange,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Object object=statement.object();
                    if (object instanceof Resource)
                        set.add(m_oimodel.getConcept((Resource)object));
                    else
                        throw new KAONException("Range cannot be a literal value.");
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of all range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    public Set getAllRangeConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getRangeConcepts());
            Iterator superProperties=getSuperProperties().iterator();
            while (superProperties.hasNext()) {
                Property property=(Property)superProperties.next();
                result.addAll(property.getAllRangeConcepts());
            }
            return result;
        }
    }
    /**
     * Checks whether supplied concept is in the range of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the range
     */
    public boolean isRangeConcept(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                return getModel().contains(m_resource,m_oimodel.m_resourceRange,conceptID);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model in which the supplied concept was added as the range to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the range to this property
     */
    public OIModel getRangeConceptOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                Statement statement=getModel().getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceRange,conceptID);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if supplied concept was added as the range to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the range concepts wihtin the <code>getOIModel()</code> model
     */
    public boolean isRangeConceptDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptID=getResourceForURI(concept.getURI());
                return getModel().thisContains(m_resource,m_oimodel.m_resourceRange,conceptID);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns whether this property is an attribute.
     *
     * @return                          <code>true</code> if property is a literal
     */
    public boolean isAttribute() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Statement rangeIsLiteral=model.getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceRange,m_oimodel.m_resourceLiteral);
                return model.contains(rangeIsLiteral);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of direct subproperties of this property.
     *
     * @return                          set of subproperties (may be empty)
     */
    public Set getSubProperties() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_oimodel.m_resourceSubPropertyOf,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(m_oimodel.getProperty(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of all subproperties of this property.
     *
     * @return                          set of all subproperties (may be empty)
     */
    public Set getAllSubProperties() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set subProperties=getSubProperties();
            Set result=new HashSet(subProperties);
            Iterator iterator=subProperties.iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                result.addAll(property.getAllSubProperties());
            }
            return result;
        }
    }
    /**
     * Tests if this property is a direct subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a direct subproperty
     */
    public boolean isDirectSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource superPropertyResource=getResourceForURI(potentialSuperProperty.getURI());
                return getModel().contains(m_resource,m_oimodel.m_resourceSubPropertyOf,superPropertyResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model containing the superproperty - subproperty association.
     *
     * @param superProperty             the superproperty of this property
     * @return                          the OI-model in which the super-subproperty association is declared
     */
    public OIModel getSuperSubPropertyOIModel(Property superProperty) throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Resource superPropertyResource=getResourceForURI(superProperty.getURI());
                Statement statement=model.getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceSubPropertyOf,superPropertyResource);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this property has been declared a subproperty of given property in this OI-model.
     *
     * @param superProperty             the superproperty of this property
     * @return                          <code>true</code> if this property has been declared a subproperty of given property in this OI-model
     */
    public boolean isSuperSubPropertyDeclaredLocally(Property superProperty) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource superPropertyResource=getResourceForURI(superProperty.getURI());
                return getModel().thisContains(m_resource,m_oimodel.m_resourceSubPropertyOf,superPropertyResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Tests if this property is a subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a subproperty
     */
    public boolean isSubPropertyOf(Property potentialSuperProperty) throws KAONException {
        synchronized (getLock()) {
            // this may be implemented more efficiently...
            return getAllSuperProperties().contains(potentialSuperProperty);
        }
    }
    /**
     * Returns the set of direct superproperties of this property.
     *
     * @return                          set of superproperties (may be empty)
     */
    public Set getSuperProperties() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(m_resource,m_oimodel.m_resourceSubPropertyOf,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    if (statement.object() instanceof Resource)
                        set.add(m_oimodel.getProperty((Resource)statement.object()));
                    else
                        throw new KAONException("Property can not be a subproperty of a literal value.");
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of all superproperties of this property.
     *
     * @return                          set of all superproperties (may be empty)
     */
    public Set getAllSuperProperties() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set superProperties=getSuperProperties();
            Set result=new HashSet(superProperties);
            Iterator iterator=superProperties.iterator();
            while (iterator.hasNext()) {
                Property property=(Property)iterator.next();
                result.addAll(property.getAllSuperProperties());
            }
            return result;
        }
    }
    /**
     * Returns the set of instances of this property.
     *
     * @return                          set of <code>PropertyInstance</code> objects of this property
     */
    public Set getPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_resource,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Instance sourceInstance=m_oimodel.getInstance(statement.subject());
                    Object targetValue;
                    if (statement.object() instanceof Resource)
                        targetValue=m_oimodel.getInstance((Resource)statement.object());
                    else
                        targetValue=((Literal)statement.object()).getLabel();
                    PropertyInstance propertyInstance=m_oimodel.getPropertyInstance(this,sourceInstance,targetValue);
                    set.add(propertyInstance);
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
}
