package edu.unika.aifb.kaon.apionrdf;

import java.util.Set;
import java.util.List;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.Collection;
import java.util.Map;
import java.util.Iterator;
import java.util.HashSet;
import java.util.Collections;
import java.net.URL;
import java.net.URLEncoder;
import java.net.URI;
import java.net.URISyntaxException;
import java.io.UnsupportedEncodingException;
import java.io.StringReader;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.util.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.vocabulary.*;
import edu.unika.aifb.kaon.api.util.*;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.magic.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;
import edu.unika.aifb.kaon.datalog.util.*;

import edu.unika.aifb.kaon.query.ast.*;
import edu.unika.aifb.kaon.query.parser.*;
import edu.unika.aifb.kaon.query.compiler.*;

/**
 * Implementation an OI-model based on the RDF API model implementation.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class OIModelImpl implements OIModel {
    /** The key for the engineering server extensional database. */
    protected static final String ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY="ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY";
    /** The key for the in-memory extensional database for temporary predicates. */
    protected static final String TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY="TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY";

    /** KAON connection that created this OI-model. */
    protected KAONConnectionImpl m_kaonConnection;
    /** Set of resources that should be skipped when navigating across the hierarchy. */
    protected Set m_skipSet;
    /** Set of included OI-models. */
    protected Set m_includedOIModels;
    /** The set of all OI-models that include this model. */
    protected Set m_allIncludedByOIModels;
    /** Event manager for this OI-model. */
    protected OIModelEventManager m_eventManager;
    /** RDF model that serves as a base for this implementation. */
    protected Model m_model;
    /** The visitor for events of included OI-models. */
    protected IncludedOIModelEventCopier m_includedOIModelEventCopier;
    /** Map of concept objects. */
    protected Map m_concepts;
    /** Map of property objects. */
    protected Map m_properties;
    /** Map of lexical entries. */
    protected Map m_lexicalEntries;
    /** Map of property instance objects. */
    protected Map m_propertyInstances;
    /** Set to <code>true</code> if the OI-model has been changed since the last save. */
    protected boolean m_hasUnsavedChanges;
    /** The datalog manager for this OI-model. */
    protected DatalogManager m_datalogManager;
    /** The predicate factory. */
    protected PredicateFactory m_predicateFactory;
    /** The bookmark for this OI-model while model is being modified. */
    protected Object m_bookmark;
    /** Resource for the class. */
    public final Resource m_resourceClass;
    /** Resource for the subClassOf. */
    public final Resource m_resourceSubClassOf;
    /** Resource for the property. */
    public final Resource m_resourceProperty;
    /** Resource for the subPropertyOf. */
    public final Resource m_resourceSubPropertyOf;
    /** Resource for the domain. */
    public final Resource m_resourceDomain;
    /** Resource for the range. */
    public final Resource m_resourceRange;
    /** Resource for the inverse. */
    public final Resource m_resourceInverse;
    /** Resource for the symmetric. */
    public final Resource m_resourceSymmetric;
    /** Resource for the transitive. */
    public final Resource m_resourceTransitive;
    /** Resource for the literal. */
    public final Resource m_resourceLiteral;
    /** Resource for the instanceOf. */
    public final Resource m_resourceInstanceOf;
    /** Resource for the minCardinality. */
    public final Resource m_resourceMinCardinality;
    /** Resource for the maxCardinality. */
    public final Resource m_resourceMaxCardinality;
    /** Literal for "true". */
    public final edu.unika.aifb.rdf.api.model.Literal m_literalTrue;
    /** Literal for "0". */
    public final edu.unika.aifb.rdf.api.model.Literal m_literalZero;
    /** Literal for "infinity". */
    public final edu.unika.aifb.rdf.api.model.Literal m_literalInfinity;

    /**
     * Creates an instance of this class and attaches it to an RDF model.
     *
     * @param kaonConnection            KAON connection that created this model
     * @param model                     RDF model that this class works with
     * @param includedOIModels          array of included OI-models
     */
    protected OIModelImpl(KAONConnectionImpl kaonConnection,Model model,OIModelImpl[] includedOIModels) throws KAONException {
        m_kaonConnection=kaonConnection;
        m_model=model;
        if (getLogicalURI()==null)
            throw new KAONException("OI-model doesn't have a logical URI - this is due to the fact that the RDF model doesn't have a logical URI.");
        m_eventManager=new OIModelEventManager(this);
        m_includedOIModelEventCopier=new IncludedOIModelEventCopier();
        m_skipSet=new HashSet();
        m_resourceClass=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getClass_());
        m_resourceSubClassOf=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getSubClassOf());
        m_resourceProperty=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getProperty());
        m_resourceSubPropertyOf=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getSubPropertyOf());
        m_resourceDomain=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getDomain());
        m_resourceRange=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getRange());
        m_resourceInverse=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getInverse());
        m_resourceSymmetric=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getSymmetric());
        m_resourceTransitive=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getTransitive());
        m_resourceLiteral=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getLiteral());
        m_resourceInstanceOf=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getInstanceOf());
        m_resourceMinCardinality=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getMinCardinality());
        m_resourceMaxCardinality=getResourceForURI(KAONVocabularyAdaptor.INSTANCE.getMaxCardinality());
        try {
            m_literalTrue=m_model.getNodeFactory().createLiteral("true");
            m_literalZero=m_model.getNodeFactory().createLiteral("0");
            m_literalInfinity=m_model.getNodeFactory().createLiteral(String.valueOf(Integer.MAX_VALUE));
        }
        catch (ModelException error) {
            throw new KAONException("RDF error",error);
        }
        m_skipSet.add(m_resourceProperty);
        m_skipSet.add(m_resourceClass);
        m_skipSet.add(m_resourceDomain);
        m_skipSet.add(m_resourceRange);
        m_skipSet.add(m_resourceInstanceOf);
        m_datalogManager=new DatalogManager();
        m_datalogManager.registerExtensionalDatabase(ENGINEERING_SERVER_EXTENSIONAL_DATABASE_KEY,new ExtensionalDatabaseImpl(m_model));
        m_datalogManager.registerExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY,new MemoryExtensionalDatabase());
        m_predicateFactory=new PredicateFactory();
        m_concepts=new WeakValueHashMap();
        m_properties=new WeakValueHashMap();
        m_lexicalEntries=new WeakValueHashMap();
        m_propertyInstances=new WeakValueHashMap();
        m_includedOIModels=new HashSet();
        m_allIncludedByOIModels=new HashSet();
        for (int i=0;i<includedOIModels.length;i++)
            addIncludedOIModel(includedOIModels[i]);
        if (!KAONConnection.ROOT_OIMODEL_URI.equals(getLogicalURI()) && getIncludedOIModel(KAONConnection.ROOT_OIMODEL_URI)==null)
            throw new KAONException("Each OI-model must include the '"+KAONConnection.ROOT_OIMODEL_URI+"' OI-model.");
    }
    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    public int getCapabilities() throws KAONException {
        synchronized (getLock()) {
            try {
                return (m_model.isPersistent() ?  CAPABILITY_ALWAYS_SAVED : 0) | CAPABILITY_SUPPORTS_NOTIFICATIONS;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the KAON connection of this OI-model.
     *
     * @return                          KAON connection of this OI-model
     */
    public KAONConnection getKAONConnection() {
        return m_kaonConnection;
    }
    /**
     * Returns the logical URI of this OI-model.
     *
     * @return                          logical URI of this OI-model
     */
    public String getLogicalURI() throws KAONException {
        synchronized (getLock()) {
            try {
                return m_model.getLogicalURI();
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the physical URI of this OI-model.
     *
     * @return                          physical URI of this OI-model
     */
    public String getPhysicalURI() throws KAONException {
        synchronized (getLock()) {
            try {
                return m_model.getPhysicalURI();
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of included OI-models.
     *
     * @return                          set of included OI-models
     */
    public Set getIncludedOIModels() {
        return m_includedOIModels;
    }
    /**
     * Returns the set of models that include this model.
     *
     * @return                          set of OI-models that include this model
     */
    public Set getAllIncludedByOIModels() {
        return m_allIncludedByOIModels;
    }
    /**
     * Returns an included OI-model with given URI.
     *
     * @param logicalURI                logical URI of the requested OI-model
     * @return                          OI-model with given logical URI or <code>null</code> if OI-model with this logical URI wasn't included
     */
    public OIModel getIncludedOIModel(String logicalURI) throws KAONException {
        synchronized (getLock()) {
            if (logicalURI.equals(getLogicalURI()))
                return this;
            Iterator iterator=m_includedOIModels.iterator();
            while (iterator.hasNext()) {
                OIModel oimodel=(OIModel)iterator.next();
                OIModel includedModel=oimodel.getIncludedOIModel(logicalURI);
                if (includedModel!=null)
                    return includedModel;
            }
            return null;
        }
    }
    /**
     * Makes an OI-model included in this model.
     *
     * @param oimodel                   OI-model that is included
     */
    public void addIncludedOIModel(OIModel oimodel) throws KAONException {
        synchronized (getLock()) {
            if (!(oimodel instanceof OIModelImpl))
                throw new KAONException("Suppled model is not a RDF model.");
            if (!m_includedOIModels.contains(oimodel)) {
                OIModelImpl includedOIModel=(OIModelImpl)oimodel;
                try {
                    m_model.addIncludedModel(includedOIModel.getModel());
                }
                catch (ModelException e) {
                    throw new KAONException("RDF exception",e);
                }
                m_includedOIModels.add(oimodel);
                List queue=new LinkedList();
                queue.add(includedOIModel);
                while (!queue.isEmpty()) {
                    OIModelImpl someOIModel=(OIModelImpl)queue.remove(0);
                    someOIModel.m_allIncludedByOIModels.add(this);
                    someOIModel.m_allIncludedByOIModels.addAll(m_allIncludedByOIModels);
                    queue.addAll(someOIModel.m_includedOIModels);
                }
            }
        }
    }
    /**
     * Makes an OI-model not included in this model.
     *
     * @param oimodel                   OI-model that is removed from the list of included models
     */
    public void removeIncludedOIModel(OIModel oimodel) throws KAONException {
        synchronized (getLock()) {
            if (!(oimodel instanceof OIModelImpl))
                throw new KAONException("Suppled model is not a RDF model.");
            if (m_includedOIModels.contains(oimodel)) {
                OIModelImpl includedOIModel=(OIModelImpl)oimodel;
                try {
                    m_model.removeIncludedModel(includedOIModel.getModel());
                }
                catch (ModelException e) {
                    throw new KAONException("RDF exception",e);
                }
                m_includedOIModels.remove(oimodel);
                // first compute the set of models that will still be the parents of the included model
                Set remainingModels=new HashSet();
                Iterator iterator=m_kaonConnection.getOpenOIModels().iterator();
                while (iterator.hasNext()) {
                    OIModelImpl someOIModel=(OIModelImpl)iterator.next();
                    if (someOIModel.m_includedOIModels.contains(includedOIModel)) {
                        remainingModels.add(someOIModel);
                        remainingModels.addAll(someOIModel.m_allIncludedByOIModels);
                    }
                }
                // now compute the models that are to be removed (this is the difference between includedOIModel.m_allIncludedByOIModels and models)
                Set modelsToRemove=new HashSet(includedOIModel.m_allIncludedByOIModels);
                modelsToRemove.removeAll(remainingModels);
                // now subtract these models from all submodels
                List queue=new LinkedList();
                queue.add(includedOIModel);
                while (!queue.isEmpty()) {
                    OIModelImpl someOIModel=(OIModelImpl)queue.remove(0);
                    someOIModel.m_allIncludedByOIModels.removeAll(modelsToRemove);
                    queue.addAll(someOIModel.m_includedOIModels);
                }
            }
        }
    }
    /**
     * Adds a listener to this OI-model.
     *
     * @param listener                  listener to be added
     */
    public void addOIModelListener(OIModelListener listener) {
        synchronized (getLock()) {
            m_eventManager.addOIModelListener(listener);
        }
    }
    /**
     * Removes a listener from this OI-model.
     *
     * @param listener                  listener to be removed
     */
    public void removeOIModelListener(OIModelListener listener) {
        synchronized (getLock()) {
            m_eventManager.removeOIModelListener(listener);
        }
    }
    /**
     * Suspends entity pool events until {@link #resumeEvents} is called.
     */
    public void suspendEvents() {
        synchronized (getLock()) {
            m_eventManager.suspendEvents();
        }
    }
    /**
     * Resumes entity pool events.
     */
    public void resumeEvents() {
        synchronized (getLock()) {
            m_eventManager.resumeEvents();
        }
    }
    /**
     * Processes changes in the change list.
     *
     * @param changeList                list of changes to the model
     */
    public void applyChanges(List changeList) throws KAONException {
        synchronized (getLock()) {
            Iterator iterator=changeList.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.setOIModel(this);
            }
            m_kaonConnection.applyChanges(changeList);
        }
    }
    /**
     * RDF-based OI-model doesn't support bulk loading.
     *
     * @param objects                   objects to be loaded
     * @param loadFlag                  the flag of what to load
     */
    public void loadObjects(Collection objects,int loadFlag) {
    }
    /**
     * Executes a query and returns the set of instances or pairs matching given query.
     *
     * @param queryString               the query to be executed
     * @return                          the collection of result objects
     */
    public Collection executeQuery(String queryString) throws KAONException {
        synchronized (getLock()) {
            Program program=null;
            Predicate queryPredicate=null;
            try {
                ExpressionParser parser=new ExpressionParser(new StringReader(queryString));
                parser.setResolutionURI(new URI(getLogicalURI()));
                Expression queryExpression=parser.ExpressionEOF();
                PredicateFactory predicateFactory=new PredicateFactory();
                ExpressionToDatalogCompiler compiler=createExpressionToDatalogCompiler(predicateFactory);
                StandardAxiomatization.getAxiomatization(this,predicateFactory,compiler.getRules());
                queryPredicate=compiler.compileExpression(queryExpression,null);
                program=compiler.getProgram();
            }
            catch (URISyntaxException error) {
                throw new KAONException("Invalid URI '"+error.getInput()+"'",error);
            }
            catch (CompilationException error) {
                throw new KAONException("Error compiling query",error);
            }
            catch (ParseException error) {
                throw new KAONException("Error parsing query",error);
            }
            catch (TokenMgrError error) {
                throw new KAONException("Error tokenizing the query",error);
            }
            catch (UncheckedException error) {
                throw (KAONException)error.getCause();
            }
            MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)m_datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            try {
                boolean[] boundVariables=new boolean[queryPredicate.getArity()];
                MagicSetRewriting rewriting=new MagicSetRewriting(program,queryPredicate,boundVariables,StandardSIPS.INSTANCE,PredicateBindingPatterns.ALL_BINDING_PATTERNS_ALLOWED);
                m_datalogManager.createMissingPredicates(rewriting.getMagicProgram(),TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
                ProgramCompiler programCompiler=new ProgramCompiler(m_datalogManager,rewriting.getMagicProgram(),Collections.EMPTY_MAP,queryPredicate);
                Evaluator evaluator=programCompiler.getEvaluator();
                Constant[] boundValues=new Constant[queryPredicate.getArity()];
                QueryOperator operator=rewriting.getResultForBoundValues(m_datalogManager,boundValues);
                evaluator.execute();
                operator.start();
                List result=new LinkedList();
                while (!operator.afterLast()) {
                    Object[] tuple=operator.tuple();
                    if (tuple.length==1) {
                        Object object=tuple[0];
                        if (object instanceof Resource)
                            result.add(getInstance((Resource)object));
                        else if (object instanceof edu.unika.aifb.rdf.api.model.Literal)
                            result.add(((edu.unika.aifb.rdf.api.model.Literal)object).getLabel());
                    }
                    else if (tuple.length==2) {
                        if (tuple[0] instanceof Resource)
                            tuple[0]=getInstance((Resource)tuple[0]);
                        else if (tuple[0] instanceof edu.unika.aifb.rdf.api.model.Literal)
                            tuple[0]=((edu.unika.aifb.rdf.api.model.Literal)tuple[0]).getLabel();
                        if (tuple[1] instanceof Resource)
                            tuple[1]=getInstance((Resource)tuple[1]);
                        else if (tuple[1] instanceof edu.unika.aifb.rdf.api.model.Literal)
                            tuple[1]=((edu.unika.aifb.rdf.api.model.Literal)tuple[1]).getLabel();
                        result.add(tuple);
                    }
                    operator.next();
                }
                operator.stop();
                return result;
            }
            catch (ModelException error) {
                throw new KAONException("RDF error",error);
            }
            catch (DatalogException error) {
                throw new KAONException("Error processing datalog query",error);
            }
            finally {
                temporaryExtensionalDatabase.clear();
            }
        }
    }
    /**
     * Creates a compiler for the query.
     *
     * @param predicateFactory          the predicate factory
     * @return                          the compiler for the query
     */
    public ExpressionToDatalogCompiler createExpressionToDatalogCompiler(PredicateFactory predicateFactory) {
        return new ExpressionToDatalogCompiler(predicateFactory) {
            protected Constant entityURIConstant(String entityURI) {
                try {
                    return new Constant(getResourceForURI(entityURI));
                }
                catch (KAONException error) {
                    throw new UncheckedException("KAON error",error);
                }
            }
            protected Constant literalConstant(Object literal) {
                try {
                    if (literal==null)
                        return new Constant();
                    else
                        return new Constant(m_model.getNodeFactory().createLiteral(literal.toString()));
                }
                catch (ModelException error) {
                    throw new UncheckedException("RDF error",error);
                }
            }
            protected void loadPropertyTypes(Set propertyNames) throws CompilationException {
                try {
                    Iterator iterator=propertyNames.iterator();
                    while (iterator.hasNext()) {
                        String propertyName=(String)iterator.next();
                        Resource propertyResource=m_model.getNodeFactory().createResource(propertyName);
                        m_propertyTypes.put(propertyName,m_model.contains(propertyResource,m_resourceRange,m_resourceLiteral) ? Boolean.TRUE : Boolean.FALSE);
                    }
                }
                catch (ModelException error) {
                    throw new CompilationException("RDF error",error);
                }
            }
        };
    }
    /**
     * Returns the underlying model.
     *
     * @return                          underlying model
     */
    public Model getModel() {
        return m_model;
    }
    /**
     * Executes given query against the standard axiomatization.
     *
     * @param queryPredicateName        the name of the answer predicate
     * @param boundValues               the values that are bound in the query
     * @return                          the operator returning the query result
     */
    public QueryOperator executeQuery(String queryPredicateName,Constant[] boundValues) throws KAONException {
        synchronized (getLock()) {
            MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)m_datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            try {
                StringBuffer buffer=new StringBuffer(queryPredicateName);
                buffer.append('_');
                for (int i=0;i<boundValues.length;i++)
                    buffer.append(boundValues[i]==null ? 'f' : 'b');
                String evaluatorKey=buffer.toString();
                Evaluator evaluator=m_datalogManager.getEvaluator(evaluatorKey);
                if (evaluator==null) {
                    Predicate queryPredicate=m_predicateFactory.getPredicate(queryPredicateName,boundValues.length);
                    boolean[] boundVariables=new boolean[boundValues.length];
                    for (int i=0;i<boundValues.length;i++)
                        boundVariables[i]=boundValues[i]!=null ? true : false;
                    List rules=new ArrayList();
                    StandardAxiomatization.getAxiomatization(this,m_predicateFactory,rules);
                    Rule[] rulesArray=new Rule[rules.size()];
                    rules.toArray(rulesArray);
                    Program originalProgram=new Program(rulesArray);
                    MagicSetRewriting rewriting=new MagicSetRewriting(originalProgram,queryPredicate,boundVariables,StandardSIPS.INSTANCE,PredicateBindingPatterns.ALL_BINDING_PATTERNS_ALLOWED);
                    m_datalogManager.createMissingPredicates(rewriting.getMagicProgram(),TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
                    ProgramCompiler programCompiler=new ProgramCompiler(m_datalogManager,rewriting.getMagicProgram(),Collections.EMPTY_MAP,queryPredicate);
                    evaluator=programCompiler.getEvaluator();
                    m_datalogManager.registerEvaluator(evaluatorKey,evaluator,rewriting);
                }
                MagicSetRewriting rewriting=m_datalogManager.getMagicSetRewriting(evaluatorKey);
                QueryOperator operator=rewriting.getResultForBoundValues(m_datalogManager,boundValues);
                evaluator.execute();
                return operator;
            }
            catch (DatalogException error) {
                temporaryExtensionalDatabase.clear();
                throw new KAONException("Datalog error",error);
            }
        }
    }
    /**
     * Should be called after query processing is done.
     */
    public void endQueryProcessing() {
        synchronized (getLock()) {
            MemoryExtensionalDatabase temporaryExtensionalDatabase=(MemoryExtensionalDatabase)m_datalogManager.getExtensionalDatabase(TEMPORARY_PREDICATES_EXTENSIONAL_DATABASE_KEY);
            temporaryExtensionalDatabase.clear();
        }
    }
    /**
     * Returns the event manager of this model.
     *
     * @return                          the event manager of this model
     */
    public OIModelEventManager getEventManager() {
        return m_eventManager;
    }
    /**
     * Returns a resource with given URI. Resource is not added to the model.
     *
     * @param uri                       URI for given resource
     * @return                          resource with given URI
     */
    public Resource getResourceForURI(String uri) throws KAONException {
        try {
            return m_model.getNodeFactory().createResource(uri);
        }
        catch (ModelException e) {
            throw new KAONException("Error creating resource with URI '"+uri+"'.",e);
        }
    }
    /**
     * Returns a concept with given URI.
     *
     * @param uri                       URI of the requested concept
     * @return                          concept with given URI
     */
    public Concept getConcept(String uri) throws KAONException {
        synchronized (getLock()) {
            return getConcept(getResourceForURI(uri));
        }
    }
    /**
     * Returns a concept for given resource.
     *
     * @param resource                  resource of the requested concept
     * @return                          concept for given resource
     */
    public Concept getConcept(Resource resource) throws KAONException {
        synchronized (getLock()) {
            try {
                Concept concept=(Concept)m_concepts.get(resource.getURI());
                if (concept==null) {
                    concept=new ConceptImpl(this,resource);
                    m_concepts.put(resource.getURI(),concept);
                }
                return concept;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the root concepts of the ontology. Root concepts are concepts that are not subclasses of any other concept.
     *
     * @return                          Set of root concepts (may be empty)
     */
    public Concept getRootConcept() throws KAONException {
        synchronized (getLock()) {
            return getConcept(KAONVocabularyAdaptor.INSTANCE.getRoot());
        }
    }
    /**
     * Returns all concepts of this ontology.
     *
     * @return                          array of all concepts in this ontology
     */
    public Set getConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_resourceInstanceOf,m_resourceClass);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(getConcept(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns a property with given URI.
     *
     * @param uri                       URI of the property
     * @return                          property with given URI
     */
    public Property getProperty(String uri) throws KAONException {
        synchronized (getLock()) {
            return getProperty(getResourceForURI(uri));
        }
    }
    /**
     * Returns a property with given URI.
     *
     * @param resource                  resource of the property
     * @return                          property with given resource
     */
    public Property getProperty(Resource resource) throws KAONException {
        synchronized (getLock()) {
            try {
                Property property=(Property)m_properties.get(resource.getURI());
                if (property==null) {
                    property=new PropertyImpl(this,resource);
                    m_properties.put(resource.getURI(),property);
                }
                return property;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all properties of this ontology.
     *
     * @return                          set of all relations of this ontology (may be empty)
     */
    public Set getProperties() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_resourceInstanceOf,m_resourceProperty);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(getProperty(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns an instance with given URI.
     *
     * @param uri                       URI of the requested instance
     * @return                          instance with given URI
     */
    public Instance getInstance(String uri) throws KAONException {
        synchronized (getLock()) {
            return getLexicalEntry(getResourceForURI(uri));
        }
    }
    /**
     * Returns an instance for given resource.
     *
     * @param resource                  resource of the requested instance
     * @return                          instance for given resource
     */
    public Instance getInstance(Resource resource) throws KAONException {
        synchronized (getLock()) {
            return getLexicalEntry(resource);
        }
    }
    /**
     * Returns all intances of this ontology.
     *
     * @return                          set of all instances of this ontology (may be empty)
     */
    public Set getInstances() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_resourceInstanceOf,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(getInstance(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns a lexical entry with given URI.
     *
     * @param uri                       URI of the lexical entry
     * @return                          lexical entry with given URI
     */
    public LexicalEntry getLexicalEntry(String uri) throws KAONException {
        synchronized (getLock()) {
            return getLexicalEntry(getResourceForURI(uri));
        }
    }
    /**
     * Returns a lexical entry with given URI.
     *
     * @param resource                  resource of the requested lexical entry
     * @return                          lexical entry with given URI
     */
    public LexicalEntry getLexicalEntry(Resource resource) throws KAONException {
        synchronized (getLock()) {
            try {
                LexicalEntry lexicalEntry=(LexicalEntry)m_lexicalEntries.get(resource.getURI());
                if (lexicalEntry==null) {
                    lexicalEntry=new LexicalEntryImpl(this,resource);
                    m_lexicalEntries.put(resource.getURI(),lexicalEntry);
                }
                return lexicalEntry;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Factory method for property instance objects.
     *
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value (<code>Instance</code> or a <code>String</code>)
     */
    public PropertyInstance getPropertyInstance(Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        synchronized (getLock()) {
            String key;
            if (targetValue instanceof Instance)
                key=property.getURI()+"$"+sourceInstance.getURI()+"$"+((Instance)targetValue).getURI();
            else
                key=property.getURI()+"$"+sourceInstance.getURI()+"$"+targetValue.toString();
            PropertyInstance propertyInstance=(PropertyInstance)m_propertyInstances.get(key);
            if (propertyInstance==null) {
                propertyInstance=new PropertyInstanceImpl((PropertyImpl)property,(InstanceImpl)sourceInstance,targetValue);
                m_propertyInstances.put(key,propertyInstance);
            }
            return propertyInstance;
        }
    }
    /**
     * Retrurns the skip set.
     *
     * @return                          returns the skip set
     */
    public Set getSkipSet() {
        return m_skipSet;
    }
    /**
     * Creates a new URI unique wihtin this OI-model.
     *
     * @return                          an new URI unique within this OI-model
     */
    public String createNewURI() throws KAONException {
        return UniqueURIGenerator.getUniqueURI(getLogicalURI());
    }
    /**
     * Creates an URI by combining the logical URI with the given suffix.
     *
     * @param suffix                    the suffix
     * @return                          a new URI with given suffix (may not be unique)
     */
    public String getURI(String suffix) throws KAONException {
        suffix=suffix.replace(' ','-');
        try {
            suffix=URLEncoder.encode(suffix,"UTF-8");
        }
        catch (UnsupportedEncodingException shouldntHappen) {
            throw new KAONException(shouldntHappen);
        }
        String uriPrefix=getLogicalURI();
        if (!uriPrefix.endsWith("#"))
            uriPrefix+="#";
        return uriPrefix+suffix;
    }
    /**
     * Makes sure that all data of this ontology is saved.
     */
    public void save() throws KAONException {
        synchronized (getLock()) {
            try {
                if (!m_model.isPersistent())
                    RDFUtil.writeModel(m_model,new URL(m_model.getPhysicalURI()),RDFUtil.DEFAULT_ENCODING);
                m_hasUnsavedChanges=false;
            }
            catch (Exception e) {
                throw new KAONException("Error saving model",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if the OI-model has been changed since the last save.
     *
     * @return                          <code>true</code> if the OI-model has been changed since the last save
     */
    public boolean hasUnsavedChanges() {
        return m_hasUnsavedChanges;
    }
    /**
     * Deletes this OI-model.
     */
    public void delete() throws KAONException {
        synchronized (getLock()) {
            try {
                RDFManager.deleteModel(m_model.getPhysicalURI());
            }
            catch (ModelException e) {
                throw new KAONException("Error deleting model",e);
            }
            m_kaonConnection.notifyOIModelDeleted(this);
            Iterator iterator=m_kaonConnection.getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelImpl someOIModel=(OIModelImpl)iterator.next();
                if (someOIModel.m_includedOIModels.contains(this))
                    someOIModel.removeIncludedOIModel(this);
            }
            m_eventManager.notifyDelete();
        }
    }
    /**
     * Refreshes the contents of this OI-model.
     */
    public void refresh() {
        synchronized (getLock()) {
            m_eventManager.notifyRefresh();
        }
    }
    /**
     * Returns the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @return                          the value (or <code>null</code> if there is no attribute with given key)
     */
    public String getAttribute(String key) throws KAONException {
        synchronized (getLock()) {
            try {
                return m_model.getAttribute(key);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Sets the value of the OI-model attribute with given key.
     *
     * @param key                       the key
     * @param value                     the value (or <code>null</code> if the attribute should be deleted)
     */
    public void setAttribute(String key,String value) throws KAONException {
        synchronized (getLock()) {
            try {
                m_model.setAttribute(key,value);
                m_hasUnsavedChanges=true;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the map of all key-value pairs of this OI-model.
     *
     * @return                          the map of key-value keys
     */
    public Map getAttributes() throws KAONException {
        synchronized (getLock()) {
            try {
                return m_model.getAttributes();
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Notifies the model that the changes will be performed.
     */
    public void startModifying() {
        suspendEvents();
        m_bookmark=m_eventManager.getBookmark();
    }
    /**
     * Notifies this OI-model about a change in this OI-model.
     *
     * @param changeEvent               the change event
     */
    public void notifyOIModelChanged(ChangeEvent changeEvent) {
        m_hasUnsavedChanges=true;
        m_eventManager.notifyChangeEvent(changeEvent);
    }
    /**
     * Notifies this OI-model about a change in an included OI-model.
     *
     * @param changeEvent               the change event
     */
    public void notifyIncludedOIModelChanged(ChangeEvent changeEvent) throws KAONException {
        changeEvent.accept(m_includedOIModelEventCopier);
    }
    /**
     * Returns <code>true</code> if this model should be committed.
     *
     * @return                          <code>true</code> iof this model should be committed
     */
    public boolean shouldCommit() {
        return m_bookmark!=null;
    }
    /**
     * Comits this OI-model.
     */
    public void commit() throws KAONException {
        try {
            if (!m_model.isAutocommit())
                m_model.commit();
            m_bookmark=null;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Rolls back the model.
     */
    public void rollback() throws KAONException {
        try {
            if (m_bookmark!=null)
                m_eventManager.applyBookmark(m_bookmark);
            if (!m_model.isAutocommit())
                m_model.rollback();
            m_bookmark=null;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Notifies the model that the changes are done.
     */
    public void endModifying() {
        m_includedOIModelEventCopier.fixCauses();
        m_includedOIModelEventCopier.clear();
        m_bookmark=null;
        resumeEvents();
    }
    /**
     * Returns the lock object for this entity.
     *
     * @return                              the lock object for the entity
     */
    protected Object getLock() {
        return m_kaonConnection;
    }

    /**
     * The class that copies the events from other OI-models.
     */
    protected class IncludedOIModelEventCopier extends ChangeEventCopier {
        public IncludedOIModelEventCopier() {
            super(OIModelImpl.this);
        }
        protected void consumeEvent(ChangeEvent changeEvent) {
            m_eventManager.notifyChangeEvent(changeEvent);
        }
    }

    /**
     * A helper unchecked exception.
     */
    protected static class UncheckedException extends RuntimeException {
        public UncheckedException(String message,Throwable cause) {
            super(message,cause);
        }
    }
}
