package edu.unika.aifb.kaon.apionrdf;

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.rdf.api.util.*;
import edu.unika.aifb.rdf.mainmemory.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * Implements a KAON connection for RDF models.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class KAONConnectionImpl implements KAONConnection {
    /** Parameter specifying the RDF factory class name. */
    public static final String RDF_FACTORY="RDF_FACTORY";
    /** Parameter specifying whether KAON connections created by this connection are sharing OI-models. */
    public static final String SHARE_OI_MODELS="SHARE_OI_MODELS";
    /** The default parameters. */
    protected static final Map s_defaultParameters;
    static {
        Map parameters=new HashMap();
        parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
        s_defaultParameters=Collections.unmodifiableMap(parameters);
        // register the URI resolver
        KAONManager.registerPhysicalURIToDefaultParametersResolver(new PhysicalURIResolver());
    }

    /** The connection to which the OI-models will be attached. */
    protected KAONConnectionImpl m_ownerKAONConnection;
    /** Parameters of the connection. */
    protected Map m_parameters;
    /** The map of OI-models indexed by the logical URI opened by this connection. */
    protected Map m_oimodelsByLogicalURI;
    /** The map of OI-models indexed by the physical URI opened by this connection. */
    protected Map m_oimodelsByPhysicalURI;
    /** The node factory. */
    protected NodeFactory m_nodeFactory;
    /** The visitor for change events. */
    protected ChangeVisitor m_applyChangeVisitor;

    /**
     * Creates an instance of this class with default parameters.
     *
     * @throws KAONException                    thrown if connection cannot be created
     */
    public KAONConnectionImpl() throws KAONException {
        this(s_defaultParameters,new HashMap(),new HashMap(),new NodeFactoryImpl(),null);
    }
    /**
     * Creates an instance of this class from a map of parameters.
     *
     * @param parameters                        the map of parameters
     * @throws KAONException                    thrown if connection cannot be created
     */
    public KAONConnectionImpl(Map parameters) throws KAONException {
        this(parameters,new HashMap(),new HashMap(),new NodeFactoryImpl(),null);
    }
    /**
     * Creates an instance of this class.
     *
     * @param parameters                        the map of parameters
     * @param oimodelsByLogicalURI              the map of OI-models by logical URI
     * @param oimodelsByPhysicalURI             the map of OI-models by physical URI
     * @param nodeFactory                       the node factory
     * @param ownerKAONConnection               the owner KAON connection
     * @throws KAONException                    thrown if connection cannot be created
     */
    public KAONConnectionImpl(Map parameters,Map oimodelsByLogicalURI,Map oimodelsByPhysicalURI,NodeFactory nodeFactory,KAONConnectionImpl ownerKAONConnection) throws KAONException {
        m_parameters=new HashMap(parameters);
        m_oimodelsByLogicalURI=oimodelsByLogicalURI;
        m_oimodelsByPhysicalURI=oimodelsByPhysicalURI;
        m_nodeFactory=nodeFactory;
        m_ownerKAONConnection=ownerKAONConnection;
        if (m_ownerKAONConnection==null)
            m_ownerKAONConnection=this;
        try {
            String rdfFactory=(String)m_parameters.get(RDF_FACTORY);
            if (rdfFactory!=null)
                RDFManager.registerFactory(rdfFactory);
        }
        catch (ModelException e) {
            throw new KAONException("Error registering the RDF factory.");
        }
        m_applyChangeVisitor=new ApplyChangeVisitor();
    }
    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    public synchronized int getCapabilities() {
        return OIModel.CAPABILITY_SUPPORTS_NOTIFICATIONS;
    }
    /**
     * Returns the parameters of this connection.
     *
     * @return                                  the parameters of this connection
     */
    public synchronized Map getParameters() {
        return m_parameters;
    }
    /**
     * Closes this connection.
     */
    public synchronized void close() {
        m_oimodelsByLogicalURI=null;
        m_oimodelsByPhysicalURI=null;
    }
    /**
     * Returns <code>true</code> if this connection is open.
     *
     * @return                                  <code>true</code> if the connection is open
     */
    public synchronized boolean isOpen() {
        return m_oimodelsByLogicalURI!=null;
    }
    /**
     * Returns a child connection. This can be used to implement advanced features, such as connection pooling.
     *
     * @return                                  a connection that is the child of this connection
     * @throws KAONException                    thrown if child connection cannot be created
     */
    public synchronized KAONConnection getConnection() throws KAONException {
        Map childOIModelsByLogicalURI=m_oimodelsByLogicalURI;
        Map childOIModelsByPhysicalURI=m_oimodelsByPhysicalURI;
        NodeFactory childNodeFactory=m_nodeFactory;
        String shareOIModels=(String)m_parameters.get(SHARE_OI_MODELS);
        KAONConnectionImpl ownerKAONConnection=m_ownerKAONConnection;
        if (!"true".equalsIgnoreCase(shareOIModels)) {
            childOIModelsByLogicalURI=new HashMap();
            childOIModelsByPhysicalURI=new HashMap();
            childNodeFactory=new NodeFactoryImpl();
            ownerKAONConnection=null;
        }
        return new KAONConnectionImpl(m_parameters,childOIModelsByLogicalURI,childOIModelsByPhysicalURI,childNodeFactory,ownerKAONConnection);
    }
    /**
     * Opens an OI-model with given physical URI.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @return                                  the OI-model
     * @throws KAONException                    thrown if OI-model cannot be opened
     */
    public synchronized OIModel openOIModelPhysical(String physicalURI) throws KAONException {
        OIModelImpl oimodel=(OIModelImpl)m_oimodelsByPhysicalURI.get(physicalURI);
        if (oimodel==null) {
            RDFManager.ModelInfo modelInfo;
            try {
                modelInfo=RDFManager.openModelEx(physicalURI,m_nodeFactory);
            }
            catch (ModelException e) {
                throw new KAONException("Cannot open RDF model with physical URI '"+physicalURI+"'",e);
            }
            oimodel=getOIModelForRDFModel(modelInfo.m_model);
            for (int i=0;i<modelInfo.m_includedModels.length;i++) {
                OIModel includedOIModel=null;
                String includedLogicalURI=modelInfo.m_includedModels[i][0];
                String includedPhysicalURI=modelInfo.m_includedModels[i][1];
                if (includedLogicalURI!=null) {
                    includedOIModel=(OIModel)m_oimodelsByLogicalURI.get(includedLogicalURI);
                    if (includedOIModel==null)
                        includedOIModel=KAONManager.resolveLogicalURI(this,includedLogicalURI);
                }
                if (includedOIModel==null && includedPhysicalURI!=null)
                    includedOIModel=openOIModelPhysical(includedPhysicalURI);
                if (includedOIModel==null)
                    throw new KAONException("Cannot resolve inclusion with logical URI '"+(includedLogicalURI==null ? "" : includedLogicalURI)+"' and physical URI '"+(includedPhysicalURI==null ? "" : includedPhysicalURI)+"'.");
                oimodel.addIncludedOIModel(includedOIModel);
            }
        }
        return oimodel;
    }
    /**
     * Opens an OI-model with given physical URI from given input stream.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param inputStream                       the RDF input stream
     * @param modelConsumer                     the model consumer - may be <code>null</code>, but may also be used to transform RDF data during parsing
     * @return                                  the OI-model
     * @throws KAONException                    thrown if OI-model cannot be opened
     */
    public synchronized OIModel openOIModelPhysical(String physicalURI,InputStream inputStream,ModelConsumer modelConsumer) throws KAONException {
        OIModelImpl oimodel=(OIModelImpl)m_oimodelsByPhysicalURI.get(physicalURI);
        if (oimodel==null) {
            RDFManager.ModelInfo modelInfo;
            try {
                modelInfo=RDFFactoryImpl.INSTANCE.openModelEx(physicalURI,m_nodeFactory,inputStream,modelConsumer);
            }
            catch (ModelException e) {
                throw new KAONException("Cannot open RDF model with physical URI '"+physicalURI+"'",e);
            }
            oimodel=getOIModelForRDFModel(modelInfo.m_model);
            for (int i=0;i<modelInfo.m_includedModels.length;i++) {
                OIModel includedOIModel=null;
                String includedLogicalURI=modelInfo.m_includedModels[i][0];
                String includedPhysicalURI=modelInfo.m_includedModels[i][1];
                if (includedLogicalURI!=null) {
                    includedOIModel=(OIModel)m_oimodelsByLogicalURI.get(includedLogicalURI);
                    if (includedOIModel==null)
                        includedOIModel=KAONManager.resolveLogicalURI(this,includedLogicalURI);
                }
                if (includedOIModel==null && includedPhysicalURI!=null)
                    includedOIModel=openOIModelPhysical(includedPhysicalURI);
                if (includedOIModel==null)
                    throw new KAONException("Cannot resolve inclusion with logical URI '"+(includedLogicalURI==null ? "" : includedLogicalURI)+"' and physical URI '"+(includedPhysicalURI==null ? "" : includedPhysicalURI)+"'.");
                oimodel.addIncludedOIModel(includedOIModel);
            }
        }
        return oimodel;
    }
    /**
     * Opens an OI-model with given logical URI.
     *
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the OI-model
     * @throws KAONException                    thrown if OI-model cannot be opened
     */
    public synchronized OIModel openOIModelLogical(String logicalURI) throws KAONException {
        OIModel oimodel=(OIModel)m_oimodelsByLogicalURI.get(logicalURI);
        if (oimodel==null)
            oimodel=KAONManager.resolveLogicalURI(this,logicalURI);
        if (oimodel==null)
            throw new KAONException("Logical URI '"+logicalURI+"' cannot be resolved.");
        return oimodel;
    }
    /**
     * Creates an OI-model with given physical and logical URIs.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param logicalURI                        the logical URI of the OI-model
     * @return                                  the OI-model
     * @throws KAONException                    thrown if OI-model cannot be opened
     */
    public synchronized OIModel createOIModel(String physicalURI,String logicalURI) throws KAONException {
        try {
            URI physicalURIObject=new URI(physicalURI);
            URI logicalURIObject=new URI(logicalURI);
            logicalURI=physicalURIObject.resolve(logicalURIObject).toString();
            if (m_oimodelsByPhysicalURI.containsKey(physicalURI))
                throw new KAONException("Model with physical URI '"+physicalURI+"' has already been opened.");
            if (m_oimodelsByLogicalURI.containsKey(logicalURI))
                throw new KAONException("Model with logical URI '"+logicalURI+"' has already been opened.");
            Model model=RDFManager.createModel(physicalURI,m_nodeFactory);
            model.setLogicalURI(logicalURI);
            OIModelImpl oimodel=getOIModelForRDFModel(model);
            oimodel.m_hasUnsavedChanges=true;
            return oimodel;
        }
        catch (ModelException e) {
            throw new KAONException("Cannot create RDF model with physical URI '"+physicalURI+"'",e);
        }
        catch (URISyntaxException e) {
            throw new KAONException("Invalid URI '"+e.getInput(),e);
        }
    }
    /**
     * Creates a new OI-model with supplied logical and physical URI and makes it an exact duplicate of supplied OI-model.
     *
     * @param sourceOIModel             the source OI-model
     * @param physicalURI               the physical URI of the new model
     * @param logicalURI                the logical URI of the new model
     * @return                          newly created OI-model
     * @throws KAONException                    thrown if OI-model cannot be opened
     */
    public synchronized OIModel createDuplicate(OIModel sourceOIModel,String physicalURI,String logicalURI) throws KAONException {
        if (m_oimodelsByPhysicalURI.containsKey(physicalURI))
            throw new KAONException("Model with physical URI '"+physicalURI+"' has already been opened.");
        if (m_oimodelsByLogicalURI.containsKey(logicalURI))
            throw new KAONException("Model with logical URI '"+logicalURI+"' has already been opened.");
        if (!(sourceOIModel instanceof OIModelImpl))
            throw new KAONException("OI-model passed in is not an RDF-based OI-model.");
        try {
            Model sourceModel=((OIModelImpl)sourceOIModel).getModel();
            Model newModel=RDFManager.createModel(physicalURI,m_nodeFactory);
            newModel.setLogicalURI(logicalURI);
            Iterator includedModels=sourceModel.getIncludedModels().iterator();
            while (includedModels.hasNext()) {
                Model includedModel=(Model)includedModels.next();
                newModel.addIncludedModel(includedModel);
            }
            if (newModel.supportsTransactions())
                newModel.setAutocommit(false);
            NodeFactory nodeFactory=newModel.getNodeFactory();
            Iterator statements=sourceModel.thisIterator();
            while (statements.hasNext()) {
                Statement statement=(Statement)statements.next();
                Resource newSubject=nodeFactory.createResource(statement.subject().getURI());
                Resource newPredicate=nodeFactory.createResource(statement.predicate().getURI());
                RDFNode newObject;
                if (statement.object() instanceof Literal)
                    newObject=nodeFactory.createLiteral(((Literal)statement.object()).getLabel());
                else
                    newObject=nodeFactory.createResource(((Resource)statement.object()).getURI());
                Statement newStatement=nodeFactory.createStatement(newSubject,newPredicate,newObject);
                newModel.add(newStatement);
            }
            if (newModel.supportsTransactions())
                newModel.commit();
            OIModelImpl oimodel=getOIModelForRDFModel(newModel);
            oimodel.m_hasUnsavedChanges=true;
            return oimodel;
        }
        catch (ModelException e) {
            throw new KAONException("Error duplicating model",e);
        }
    }
    /**
     * Notifies the connection that the OI-model has been deleted.
     *
     * @param oimodel                           OI-model that is removed
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void notifyOIModelDeleted(OIModel oimodel) throws KAONException {
        m_oimodelsByLogicalURI.remove(oimodel.getLogicalURI());
        m_oimodelsByPhysicalURI.remove(oimodel.getPhysicalURI());
    }
    /**
     * Returns an OI-model for given RDF model.
     *
     * @param model                             the RDF model
     * @return                                  OI-model for given RDF model
     * @throws KAONException                    thrown if OI-model cannot be obtained
     */
    public synchronized OIModelImpl getOIModelForRDFModel(Model model) throws KAONException {
        try {
            OIModelImpl oimodel=(OIModelImpl)m_oimodelsByLogicalURI.get(model.getLogicalURI());
            if (oimodel!=null) {
                if (oimodel.getModel()!=model)
                    throw new KAONException("Another model with logical URI equal to logical URI of the supplied model has already been opened.");
            }
            else {
                oimodel=(OIModelImpl)m_oimodelsByPhysicalURI.get(model.getPhysicalURI());
                if (oimodel!=null) {
                    if (oimodel.getModel()!=model)
                        throw new KAONException("Another model with physical URI equal to the physical URI of the supplied model has already been opened.");
                }
                else {
                    if (model.supportsTransactions())
                        model.setAutocommit(false);
                    if (!ROOT_OIMODEL_URI.equals(model.getLogicalURI())) {
                        OIModelImpl rootOIModel=(OIModelImpl)openOIModelLogical(ROOT_OIMODEL_URI);
                        model.addIncludedModel(rootOIModel.getModel());
                    }
                    List models=new ArrayList();
                    Iterator iterator=model.getIncludedModels().iterator();
                    while (iterator.hasNext()) {
                        Model includedModel=(Model)iterator.next();
                        models.add(getOIModelForRDFModel(includedModel));
                    }
                    OIModelImpl[] includedOIModels=new OIModelImpl[models.size()];
                    models.toArray(includedOIModels);
                    oimodel=new OIModelImpl(m_ownerKAONConnection,model,includedOIModels);
                    m_oimodelsByLogicalURI.put(oimodel.getLogicalURI(),oimodel);
                    m_oimodelsByPhysicalURI.put(oimodel.getPhysicalURI(),oimodel);
                }
            }
            return oimodel;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns an OI-model for given RDF model, but throws an exception if the model doesn't exist.
     *
     * @param model                             the RDF model
     * @return                                  the OI-model for given RDF model
     * @throws KAONException                    thrown if OI-model cannot be obtained
     */
    public synchronized OIModel getOIModelForRDFModelMustExist(Model model) throws KAONException {
        try {
            OIModel oimodel=(OIModel)m_oimodelsByLogicalURI.get(model.getLogicalURI());
            if (oimodel==null)
                oimodel=(OIModel)m_oimodelsByPhysicalURI.get(model.getPhysicalURI());
            if (oimodel==null)
                throw new KAONException("RDF model hasn't been opened by the KAON connection.");
            return oimodel;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns the set of all open OI-model objects.
     *
     * @return                              the set of all open OI-model objects
     */
    public synchronized Set getOpenOIModels() {
        return new HashSet(m_oimodelsByLogicalURI.values());
    }
    /**
     * Returns the set of OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     */
    public synchronized Set getAllOIModels() {
        return new HashSet(m_oimodelsByLogicalURI.values());
    }
    /**
     * Returns the set of logical URIs of all OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     */
    public synchronized Set getAllOIModelLogicalURIs() {
        return m_oimodelsByLogicalURI.keySet();
    }
    /**
     * Applies the list of changes to the models in this connection.
     *
     * @param changes                           list of changes to the models in the connection
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void applyChanges(List changes) throws KAONException {
        try {
            Iterator iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelImpl oimodel=(OIModelImpl)iterator.next();
                oimodel.startModifying();
            }
            iterator=changes.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                OIModelImpl oimodel=(OIModelImpl)changeEvent.getOIModel();
                changeEvent.accept(m_applyChangeVisitor);
                oimodel.notifyOIModelChanged(changeEvent);
                if (!oimodel.getAllIncludedByOIModels().isEmpty()) {
                    Iterator superOIModels=oimodel.getAllIncludedByOIModels().iterator();
                    while (superOIModels.hasNext()) {
                        OIModelImpl superOIModel=(OIModelImpl)superOIModels.next();
                        superOIModel.notifyIncludedOIModelChanged(changeEvent);
                    }
                }
            }
            iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelImpl oimodel=(OIModelImpl)iterator.next();
                if (oimodel.shouldCommit())
                    oimodel.commit();
            }
        }
        catch (KAONException e) {
            Iterator iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelImpl oimodel=(OIModelImpl)iterator.next();
                try {
                    if (oimodel.shouldCommit())
                        oimodel.rollback();
                }
                catch (KAONException ignored) {
                }
            }
            throw e;
        }
        finally {
            Iterator iterator=getOpenOIModels().iterator();
            while (iterator.hasNext()) {
                OIModelImpl oimodel=(OIModelImpl)iterator.next();
                oimodel.endModifying();
            }
        }
    }

    /**
     * The resolver for RDF physical URIs.
     */
    protected static class PhysicalURIResolver implements PhysicalURIToDefaultParametersResolver {
        public Map getDefaultParametersForPhysicalURI(String physicalURI,Map contextParameters) {
            if (physicalURI.startsWith("jar:") || physicalURI.startsWith("file:") || physicalURI.startsWith("http:"))
                return new HashMap(s_defaultParameters);
            else
                return null;
        }
    }
}
