package edu.unika.aifb.kaon.apionrdf;

import java.util.Iterator;
import java.util.Set;
import java.util.Map;
import java.util.HashSet;
import java.util.HashMap;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.Constant;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * Implementation of the Instance interface that relies on the RDF API as the model.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class InstanceImpl extends AbstractEntityImpl implements Instance {
    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param resource                  resource
     */
    public InstanceImpl(OIModelImpl oimodel,Resource resource) {
        super(oimodel,resource);
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        return OIModel.LOAD_INSTANCE_ALL;
    }
    /**
     * Tests whether this instance is part of its associated instance pool.
     *
     * @return                                  <code>true</code> if instance is part of the instance pool
     */
    public boolean isInOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                if (m_resource.getURI().equals(KAONVocabularyAdaptor.INSTANCE.getRoot()))
                    return false;
                if (getModel().contains(m_resource,m_oimodel.m_resourceInstanceOf,null))
                    return true;
                if (getSpanningConcept().isInOIModel())
                    return true;
                if (getSpanningProperty().isInOIModel())
                    return true;
                return false;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. Note that this may return <code>false</code> if this
     * entity was declared in an included OI-model.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            try {
                if (m_resource.getURI().equals(KAONVocabularyAdaptor.INSTANCE.getRoot()))
                    return false;
                if (getModel().thisContains(m_resource,m_oimodel.m_resourceInstanceOf,null))
                    return true;
                if (getSpanningConcept().isDeclaredLocally())
                    return true;
                if (getSpanningProperty().isDeclaredLocally())
                    return true;
                return false;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model where this entity was declared originally.
     *
     * @return                          the OI-model there this entity was declated originally
     */
    public OIModel getSourceOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                if (model.thisContains(m_resource,m_oimodel.m_resourceInstanceOf,null))
                    return m_oimodel;
                Iterator allModels=model.getAllIncludedModels().iterator();
                while (allModels.hasNext()) {
                    Model includedModel=(Model)allModels.next();
                    if (includedModel.thisContains(m_resource,m_oimodel.m_resourceInstanceOf,null))
                        return ((KAONConnectionImpl)m_oimodel.getKAONConnection()).getOIModelForRDFModelMustExist(includedModel);
                }
                OIModel oimodel=getSpanningConcept().getSourceOIModel();
                if (oimodel!=null)
                    return oimodel;
                return getSpanningProperty().getSourceOIModel();
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns set of <code>Concepts</code> objects that specify which concepts is this instance instance of.
     *
     * @return                                  set of concepts that this instance is instance of (may be empty)
     */
    public Set getParentConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                set.add(m_oimodel.getRootConcept());
                Model result=getModel().find(m_resource,m_oimodel.m_resourceInstanceOf,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    if (statement.object() instanceof Resource) {
                        Resource object=(Resource)statement.object();
                        if (!isInSkipSet(object))
                            set.add(m_oimodel.getConcept(object.getURI()));
                    }
                    else
                        throw new KAONException("Parent of a concept cannot be a literal.");
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns set of all <code>Concepts</code> objects that this instance is instance of.
     *
     * @return                                  set of all concepts that this instance is instance of (may be empty)
     */
    public Set getAllParentConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set parentConcepts=getParentConcepts();
            Set result=new HashSet(parentConcepts);
            Iterator iterator=parentConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSuperConcepts());
            }
            return result;
        }
    }
    /**
     * Tests if given concept is a direct parent of this instance.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a direct parent of this instance
     */
    public boolean isDirectParent(Concept potentialParent) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource parentResource=getResourceForURI(potentialParent.getURI());
                return getModel().contains(m_resource,m_oimodel.m_resourceInstanceOf,parentResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model containing the this concept - instance association.
     *
     * @param concept                   the concept of this instance
     * @return                          the OI-model in which the concept-instance association is declared
     */
    public OIModel getConceptInstanceOIModel(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                if (KAONVocabularyAdaptor.INSTANCE.getRoot().equals(concept.getURI()))
                    return getSourceOIModel();
                else {
                    Resource conceptResource=getResourceForURI(concept.getURI());
                    Statement statement=getModel().getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceInstanceOf,conceptResource);
                    return getOIModelForStatement(statement);
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if the supplied concept has been declared a parent of this instance in this OI-model.
     *
     * @param concept                   the concept of this instance
     * @return                          <code>true</code> if supplied concept has been declared a parent of this instance in this OI-model
     */
    public boolean isConceptInstanceDeclaredLocally(Concept concept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource conceptResource=getResourceForURI(concept.getURI());
                return getModel().thisContains(m_resource,m_oimodel.m_resourceInstanceOf,conceptResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Tests if given concept is a parent of this concept.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a parent of this instance
     */
    public boolean isParent(Concept potentialParent) throws KAONException {
        synchronized (getLock()) {
            return getAllParentConcepts().contains(potentialParent);
        }
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point out from this instance.
     *
     * @return                                  set of property instances pointing from this instance (may be empty)
     */
    public Set getFromPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(m_resource,null,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Resource predicate=statement.predicate();
                    if (!isInSkipSet(predicate)) {
                        Property property=m_oimodel.getProperty(predicate.getURI());
                        if (property.isInOIModel()) {
                            PropertyInstance propertyInstance;
                            if (statement.object() instanceof Resource)
                                propertyInstance=m_oimodel.getPropertyInstance(property,this,m_oimodel.getInstance((Resource)statement.object()));
                            else
                                propertyInstance=m_oimodel.getPropertyInstance(property,this,((Literal)statement.object()).getLabel());
                            set.add(propertyInstance);
                        }
                    }
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the map of sets of property values. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getFromPropertyValues() throws KAONException {
        synchronized (getLock()) {
            try {
                Map map=new HashMap();
                Model result=getModel().find(m_resource,null,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Resource predicate=statement.predicate();
                    if (!isInSkipSet(predicate)) {
                        Property property=m_oimodel.getProperty(predicate.getURI());
                        if (property.isInOIModel()) {
                            Set set=(Set)map.get(property);
                            if (set==null) {
                                set=new HashSet();
                                map.put(property,set);
                            }
                            if (statement.object() instanceof Resource)
                                set.add(m_oimodel.getInstance((Resource)statement.object()));
                            else
                                set.add(((Literal)statement.object()).getLabel());
                        }
                    }
                }
                return map;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the value of a given property for this instance. If there is no value for a property, <code>null</code> is
     * returned. If there are multiple values for a property, one is picked arbitrarily and returned. Returned value is either
     * an <code>Instance</code> or a <code>String</code>.
     *
     * @param property                          property
     * @return                                  value of the property or <code>null</code> if there is no value
     */
    public Object getFromPropertyValue(Property property) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource propertyID=getResourceForURI(property.getURI());
                Model result=getModel().find(m_resource,propertyID,null);
                if (result.size()==0)
                    return null;
                else {
                    Statement statement=(Statement)result.iterator().next();
                    Object value;
                    if (statement.object() instanceof Resource)
                        value=m_oimodel.getInstance((Resource)statement.object());
                    else
                        value=((Literal)statement.object()).getLabel();
                    return value;
                }
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of values of a given property. It there are no values, an empty set is returned. Elements of the set
     * may be <code>Instance</code> or <code>String</code> objects.
     *
     * @param property                          property
     * @return                                  set of values of the property
     */
    public Set getFromPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Resource propertyID=getResourceForURI(property.getURI());
                Model result=getModel().find(m_resource,propertyID,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Object value;
                    if (statement.object() instanceof Resource)
                        value=m_oimodel.getInstance((Resource)statement.object());
                    else
                        value=((Literal)statement.object()).getLabel();
                    set.add(value);
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Tests whether this instance points to given value using specified property.
     *
     * @param property                          property
     * @param targetValue                       the value (can be a string or an instance)
     * @return                                  <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     */
    public boolean pointsToValue(Property property,Object targetValue) throws KAONException {
        synchronized (getLock()) {
            try {
                boolean result;
                if (property.isAttribute()) {
                    QueryOperator operator=m_oimodel.executeQuery("AttributeInstance_a",new Constant[] { new Constant(getResourceForURI(property.getURI())),new Constant(m_resource),new Constant(targetValue) });
                    operator.start();
                    result=!operator.afterLast();
                    operator.stop();
                }
                else {
                    QueryOperator operator=m_oimodel.executeQuery("RelationInstance_a",new Constant[] { new Constant(getResourceForURI(property.getURI())),new Constant(m_resource),new Constant(getResourceForURI(((Instance)targetValue).getURI())) });
                    operator.start();
                    result=!operator.afterLast();
                    operator.stop();
                }
                return result;
            }
            catch (DatalogException error) {
                throw new KAONException("Datalog error",error);
            }
            finally {
                m_oimodel.endQueryProcessing();
            }
        }
    }
    /**
     * Returns the values of a property for this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of property values for this given property
     */
    public Set getAllFromPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            try {
                Set result=new HashSet();
                if (property.isAttribute()) {
                    QueryOperator operator=m_oimodel.executeQuery("AttributeInstance_a",new Constant[] { new Constant(getResourceForURI(property.getURI())),new Constant(m_resource),null });
                    operator.start();
                    while (!operator.afterLast()) {
                        Object[] tuple=operator.tuple();
                        if (tuple[0] instanceof Resource && tuple[1] instanceof Resource && tuple[2] instanceof Literal)
                            result.add(((Literal)tuple[2]).getLabel());
                        operator.next();
                    }
                    operator.stop();
                }
                else {
                    QueryOperator operator=m_oimodel.executeQuery("RelationInstance_a",new Constant[] { new Constant(getResourceForURI(property.getURI())),new Constant(m_resource),null });
                    operator.start();
                    while (!operator.afterLast()) {
                        Object[] tuple=operator.tuple();
                        if (tuple[0] instanceof Resource && tuple[1] instanceof Resource && tuple[2] instanceof Resource)
                            result.add(m_oimodel.getInstance((Resource)tuple[2]));
                        operator.next();
                    }
                    operator.stop();
                }
                return result;
            }
            catch (DatalogException error) {
                throw new KAONException("Datalog error",error);
            }
            catch (ModelException error) {
                throw new KAONException("RDF error",error);
            }
            finally {
                m_oimodel.endQueryProcessing();
            }
        }
    }
    /**
     * Returns the values of all properties for this instance, including the inferred ones.
     *
     * @return                                  set of all property values for this instance
     */
    public Map getAllFromPropertyValues() throws KAONException {
        synchronized (getLock()) {
            try {
                Map result=new HashMap();
                QueryOperator operator=m_oimodel.executeQuery("PropertyInstance_a",new Constant[] { null,new Constant(m_resource),null,null });
                operator.start();
                while (!operator.afterLast()) {
                    Object[] tuple=operator.tuple();
                    if (tuple[0] instanceof Resource && tuple[1] instanceof Resource && (tuple[2] instanceof Resource || tuple[3] instanceof Literal)) {
                        Property property=m_oimodel.getProperty((Resource)tuple[0]);
                        Set set=(Set)result.get(property);
                        if (set==null) {
                            set=new HashSet();
                            result.put(property,set);
                        }
                        if (tuple[2] instanceof Resource)
                            set.add(m_oimodel.getInstance((Resource)tuple[2]));
                        else if (tuple[3] instanceof Literal)
                            set.add(((Literal)tuple[3]).getLabel());
                    }
                    operator.next();
                }
                operator.stop();
                return result;
            }
            catch (DatalogException error) {
                throw new KAONException("Datalog error",error);
            }
            catch (ModelException error) {
                throw new KAONException("RDF error",error);
            }
            finally {
                m_oimodel.endQueryProcessing();
            }
        }
    }
    /**
     * Adds all elements to from source to destination that are not in the control set as well.
     *
     * @param source                            source set
     * @param destination                       destination set
     * @param control                           control set
     */
    protected void addElements(Set source,Set destination,Set control) {
        Iterator iterator=source.iterator();
        while (iterator.hasNext()) {
            Object element=iterator.next();
            if (!control.contains(element))
                destination.add(element);
        }
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point to this instance.
     *
     * @return                                  set of property instances pointing to this instance (may be empty)
     */
    public Set getToPropertyInstances() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,null,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Resource predicate=statement.predicate();
                    if (!isInSkipSet(predicate)) {
                        Property property=m_oimodel.getProperty(predicate.getURI());
                        if (property.isInOIModel()) {
                            PropertyInstance propertyInstance=m_oimodel.getPropertyInstance(property,m_oimodel.getInstance(statement.subject()),this);
                            set.add(propertyInstance);
                        }
                    }
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the map of sets of property values pointing to the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getToPropertyValues() throws KAONException {
        synchronized (getLock()) {
            try {
                Map map=new HashMap();
                Model result=getModel().find(null,null,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Resource predicate=statement.predicate();
                    if (!isInSkipSet(predicate)) {
                        Property property=m_oimodel.getProperty(predicate.getURI());
                        if (property.isInOIModel()) {
                            Set set=(Set)map.get(property);
                            if (set==null) {
                                set=new HashSet();
                                map.put(property,set);
                            }
                            set.add(m_oimodel.getInstance(statement.subject()));
                        }
                    }
                }
                return map;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of <code>Instnace</code> objects that point to this instance via given property.
     *
     * @param property                          property of interest
     * @return                                  set of <code>Instance</code> objects pointing to this instance via given property
     */
    public Set getToPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Resource propertyID=getResourceForURI(property.getURI());
                Model result=getModel().find(null,propertyID,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    Instance instance=m_oimodel.getInstance(statement.subject());
                    set.add(instance);
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the values of a property pointing to this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of all property values pointing to this instance for given property
     */
    public Set getAllToPropertyValues(Property property) throws KAONException {
        synchronized (getLock()) {
            try {
                Set result=new HashSet();
                if (!property.isAttribute()) {
                    QueryOperator operator=m_oimodel.executeQuery("RelationInstance_a",new Constant[] { new Constant(getResourceForURI(property.getURI())),null,new Constant(m_resource) });
                    operator.start();
                    while (!operator.afterLast()) {
                        Object[] tuple=operator.tuple();
                        if (tuple[0] instanceof Resource && tuple[1] instanceof Resource && tuple[2] instanceof Resource)
                            result.add(m_oimodel.getInstance((Resource)tuple[1]));
                        operator.next();
                    }
                    operator.stop();
                }
                return result;
            }
            catch (DatalogException error) {
                throw new KAONException("Datalog error",error);
            }
            finally {
                m_oimodel.endQueryProcessing();
            }
        }
    }
    /**
     * Returns the values of all properties pointing to this instance, including the inferred ones.
     *
     * @return                                  set of all property values pointing to this instance
     */
    public Map getAllToPropertyValues() throws KAONException {
        synchronized (getLock()) {
            try {
                Map result=new HashMap();
                QueryOperator operator=m_oimodel.executeQuery("PropertyInstance_a",new Constant[] { null,null,new Constant(m_resource),null });
                operator.start();
                while (!operator.afterLast()) {
                    Object[] tuple=operator.tuple();
                    if (tuple[0] instanceof Resource && tuple[1] instanceof Resource && tuple[2] instanceof Resource) {
                        Property property=m_oimodel.getProperty((Resource)tuple[0]);
                        Set set=(Set)result.get(property);
                        if (set==null) {
                            set=new HashSet();
                            result.put(property,set);
                        }
                        set.add(m_oimodel.getInstance((Resource)tuple[1]));
                    }
                    operator.next();
                }
                operator.stop();
                return result;
            }
            catch (DatalogException error) {
                throw new KAONException("Datalog error",error);
            }
            finally {
                m_oimodel.endQueryProcessing();
            }
        }
    }
    /**
     * Returns the spanning concept for this instance.
     *
     * @return                                  spanning concept for this instance
     */
    public Concept getSpanningConcept() throws KAONException {
        return m_oimodel.getConcept(getURI());
    }
    /**
     * Returns the spanning property for this instance.
     *
     * @return                                  spanning property for this instance
     */
    public Property getSpanningProperty() throws KAONException {
        return m_oimodel.getProperty(getURI());
    }
    /**
     * Returns the instance associated with this object.
     *
     * @return                              instance associated with this object
     */
    public Instance getSpanningInstance() {
        return this;
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
}
