package edu.unika.aifb.kaon.apionrdf;

import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.StringTokenizer;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import edu.unika.aifb.rdf.api.model.*;

import edu.unika.aifb.kaon.api.vocabulary.*;

import edu.unika.aifb.kaon.datalog.*;
import edu.unika.aifb.kaon.datalog.program.*;
import edu.unika.aifb.kaon.datalog.evaluation.*;

/**
 * Implements the extensional database over the RDF API.
 */
public class ExtensionalDatabaseImpl extends PredicateExtensionExtensionalDatabase {
    /** The set of URIs that are not visible in the model. */
    protected static final Set s_notVisibleURIs=new HashSet();
    static {
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getInstanceOf());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getSubClassOf());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getSubPropertyOf());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getClass_());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getDomain());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getRange());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getInverse());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getSymmetric());
        s_notVisibleURIs.add(KAONVocabularyAdaptor.INSTANCE.getTransitive());
    }

    /** The checker for the binding patterns. */
    public static final PredicateBindingPatterns PREDICATE_BINDING_PATTERNS=new PredicateBindingPatternsImpl();

    /** The RDF model for this extensional database. */
    protected Model m_model;
    /** The Triple predicate extension. */
    protected PredicateExtension m_triplePredicateExtension;
    /** The AttributeInstance predicate extension. */
    protected PredicateExtension m_regExpPredicateExtension;

    /**
     * Creates an instance of this class.
     *
     * @param model                         the RDF model
     */
    public ExtensionalDatabaseImpl(Model model) {
        m_model=model;
        m_triplePredicateExtension=new TriplePredicateExtension();
        m_regExpPredicateExtension=new RegExpPredicateExtension();
    }
    /**
     * Returns the extension for given predicate.
     *
     * @param predicate                     the predicate
     * @return                              the extension for the given predicate
     */
    public PredicateExtension getPredicateExtension(Predicate predicate) {
        String predicateFullName=predicate.getFullName();
        if ("Triple/3".equals(predicateFullName))
            return m_triplePredicateExtension;
        else if ("IsResource/1".equals(predicateFullName))
            return IsResourcePredicateExtension.INSTANCE;
        else if ("IsLiteral/1".equals(predicateFullName))
            return IsLiteralPredicateExtension.INSTANCE;
        else if ("IsVisible/1".equals(predicateFullName))
            return IsVisiblePredicateExtension.INSTANCE;
        else if ("RegExp/2".equals(predicateFullName))
            return m_regExpPredicateExtension;
        else
            return null;
    }

    protected static final Predicate TRIPLE_PREDICATE=new Predicate(true,"Triple",3);
    /**
     * The extension for the Triple predicate.
     */
    protected class TriplePredicateExtension implements PredicateExtension {

        public Predicate getPredicate() {
            return TRIPLE_PREDICATE;
        }
        public boolean getContainsDisjunctionInfo() {
            return false;
        }
        public Iterator selectTuples(Object[] values,int[] positions) throws DatalogException {
            try {
                Resource subject=null;
                Resource predicate=null;
                RDFNode object=null;
                for (int i=0;i<positions.length;i++) {
                    Object value=values[i];
                    switch (positions[i]) {
                    case 0:
                        if (!(value instanceof Resource))
                            return EmptyIterator.INSTANCE;
                        subject=(Resource)value;
                        break;
                    case 1:
                        if (!(value instanceof Resource))
                            return EmptyIterator.INSTANCE;
                        predicate=(Resource)value;
                        break;
                    case 2:
                        if (!(value instanceof RDFNode))
                            return EmptyIterator.INSTANCE;
                        object=(RDFNode)value;
                        break;
                    }
                }
                return new StatementToTripleIterator(m_model.find(subject,predicate,object).iterator());
            }
            catch (ModelException error) {
                throw new DatalogException("RDF error",error);
            }
        }
    }

    /**
     * The iterator converting statements to triples.
     */
    protected static class StatementToTripleIterator implements Iterator {
        protected Iterator m_statementIterator;

        public StatementToTripleIterator(Iterator statementIterator) {
            m_statementIterator=statementIterator;
        }
        public boolean hasNext() {
            return m_statementIterator.hasNext();
        }
        public Object next() {
            try {
                Statement statement=(Statement)m_statementIterator.next();
                return new Object[] { statement.subject(),statement.predicate(),statement.object() };
            }
            catch (ModelException ignored) {
                return new Object[3];
            }
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }

    protected static final Predicate REGEXP_PREDICATE=new Predicate(true,"RegExp",2);
    /**
     * The extension for the RegExp predicate.
     */
    protected class RegExpPredicateExtension implements PredicateExtension {

        public Predicate getPredicate() {
            return REGEXP_PREDICATE;
        }
        public boolean getContainsDisjunctionInfo() {
            return false;
        }
        public Iterator selectTuples(Object[] values,int[] positions) throws DatalogException {
            try {
                edu.unika.aifb.rdf.api.model.Literal object=null;
                edu.unika.aifb.rdf.api.model.Literal regularExpression=null;
                for (int i=0;i<positions.length;i++) {
                    Object value=values[i];
                    switch (positions[i]) {
                    case 0:
                        if (!(value instanceof edu.unika.aifb.rdf.api.model.Literal))
                            return EmptyIterator.INSTANCE;
                        object=(edu.unika.aifb.rdf.api.model.Literal)value;
                        break;
                    case 1:
                        if (!(value instanceof edu.unika.aifb.rdf.api.model.Literal))
                            return EmptyIterator.INSTANCE;
                        regularExpression=(edu.unika.aifb.rdf.api.model.Literal)value;
                        break;
                    }
                }
                if (regularExpression==null)
                    throw new DatalogException("RegExp called with disallowed binding pattern.");
                Iterator iterator;
                if (object!=null)
                    iterator=m_model.find(null,null,object).iterator();
                else
                    iterator=m_model.iterator();
                return new RegExpIterator(regularExpression,getPatterns(regularExpression.getLabel()),iterator);
            }
            catch (ModelException error) {
                throw new DatalogException("RDF error",error);
            }
        }
        protected Pattern[] getPatterns(String regularExpression) throws DatalogException {
            List patterns=new ArrayList();
            StringTokenizer tokenizer=new StringTokenizer(regularExpression);
            while (tokenizer.hasMoreTokens()) {
                String token=tokenizer.nextToken();
                StringBuffer filterExpression=new StringBuffer();
                filterExpression.append("((.*\\s+)|^)");
                int lastStarIndex=0;
                int starIndex;
                do {
                    starIndex=token.indexOf('*',lastStarIndex);
                    String subString;
                    if (starIndex!=-1)
                        subString=token.substring(lastStarIndex,starIndex);
                    else
                        subString=token.substring(lastStarIndex);
                    if (subString.length()!=0) {
                        filterExpression.append("\\Q");
                        filterExpression.append(subString);
                        filterExpression.append("\\E");
                    }
                    if (starIndex!=-1) {
                        filterExpression.append(".*");
                        lastStarIndex=starIndex+1;
                    }
                } while (starIndex!=-1);
                filterExpression.append("((\\s+.*)|$)");
                String expression=filterExpression.toString();
                try {
                    patterns.add(Pattern.compile(expression,Pattern.CASE_INSENSITIVE));
                }
                catch (PatternSyntaxException error) {
                    throw new DatalogException("Error in regular expression.");
                }
            }
            Pattern[] patternsArray=new Pattern[patterns.size()];
            patterns.toArray(patternsArray);
            return patternsArray;
        }
    }

    /**
     * The iterator filtering statements by regular expressions.
     */
    protected static class RegExpIterator implements Iterator {
        protected edu.unika.aifb.rdf.api.model.Literal m_regularExpression;
        protected Pattern[] m_patterns;
        protected Iterator m_statementIterator;
        protected Object[] m_currentTuple;

        public RegExpIterator(edu.unika.aifb.rdf.api.model.Literal regularExpression,Pattern[] patterns,Iterator statementIterator) {
            m_regularExpression=regularExpression;
            m_patterns=patterns;
            m_statementIterator=statementIterator;
            advance();
        }
        public boolean hasNext() {
            return m_currentTuple!=null;
        }
        public Object next() {
            if (m_currentTuple==null)
                throw new NoSuchElementException();
            Object[] result=m_currentTuple;
            advance();
            return result;
        }
        protected void advance() {
            while (m_statementIterator.hasNext()) {
                Statement statement=(Statement)m_statementIterator.next();
                try {
                    RDFNode object=statement.object();
                    if (object instanceof edu.unika.aifb.rdf.api.model.Literal) {
                        String value=((edu.unika.aifb.rdf.api.model.Literal)object).getLabel();
                        boolean matches=true;
                        for (int i=0;matches && i<m_patterns.length;i++)
                            if (!m_patterns[i].matcher(value).matches())
                                matches=false;
                        if (matches) {
                            m_currentTuple=new Object[] { object,m_regularExpression };
                            return;
                        }
                    }
                }
                catch (ModelException ignored) {
                }
            }
            m_currentTuple=null;
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * The extension for the IsResource predicate.
     */
    protected static class IsResourcePredicateExtension implements PredicateExtension {
        /** The only instance. */
        public static final PredicateExtension INSTANCE=new IsResourcePredicateExtension();
        protected static final Predicate PREDICATE=new Predicate(true,"IsResource",1);

        public Predicate getPredicate() {
            return PREDICATE;
        }
        public boolean getContainsDisjunctionInfo() {
            return false;
        }
        public Iterator selectTuples(Object[] values,int[] positions) throws DatalogException {
            if (positions.length!=1 || positions[0]!=0)
                throw new DatalogException("IsResource called with disallowed binding pattern.");
            Object value=values[0];
            if (value instanceof Resource)
                return new SingletonIterator(new Object[] { value });
            else
                return EmptyIterator.INSTANCE;
        }
    }

    /**
     * The extension for the IsLiteral predicate.
     */
    protected static class IsLiteralPredicateExtension implements PredicateExtension {
        /** The only instance. */
        public static final PredicateExtension INSTANCE=new IsLiteralPredicateExtension();
        protected static final Predicate PREDICATE=new Predicate(true,"IsLiteral",1);

        public Predicate getPredicate() {
            return PREDICATE;
        }
        public boolean getContainsDisjunctionInfo() {
            return false;
        }
        public Iterator selectTuples(Object[] values,int[] positions) throws DatalogException {
            if (positions.length!=1 || positions[0]!=0)
                throw new DatalogException("IsLiteral called with disallowed binding pattern.");
            Object value=values[0];
            if (value instanceof edu.unika.aifb.rdf.api.model.Literal)
                return new SingletonIterator(new Object[] { value });
            else
                return EmptyIterator.INSTANCE;
        }
    }

    /**
     * The extension for the IsVisible predicate.
     */
    protected static class IsVisiblePredicateExtension implements PredicateExtension {
        /** The only instance. */
        public static final PredicateExtension INSTANCE=new IsVisiblePredicateExtension();
        protected static final Predicate PREDICATE=new Predicate(true,"IsVisible",1);

        public Predicate getPredicate() {
            return PREDICATE;
        }
        public boolean getContainsDisjunctionInfo() {
            return false;
        }
        public Iterator selectTuples(Object[] values,int[] positions) throws DatalogException {
            try {
                if (positions.length!=1 || positions[0]!=0)
                    throw new DatalogException("IsVisible called with disallowed binding pattern.");
                Object value=values[0];
                if (value instanceof Resource) {
                    String uri=((Resource)value).getURI();
                    if (s_notVisibleURIs.contains(uri))
                        return EmptyIterator.INSTANCE;
                }
                return new SingletonIterator(new Object[] { value });
            }
            catch (ModelException error) {
                throw new DatalogException("RDF error",error);
            }
        }
    }

    /**
     * The empty iterator.
     */
    protected static class EmptyIterator implements Iterator {
        public static Iterator INSTANCE=new EmptyIterator();

        public boolean hasNext() {
            return false;
        }
        public Object next() {
            throw new NoSuchElementException();
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * The iterator over a singleton triple.
     */
    protected static class SingletonIterator implements Iterator {
        protected Object[] m_triple;

        public SingletonIterator(Object[] triple) {
            m_triple=triple;
        }
        public boolean hasNext() {
            return m_triple!=null;
        }
        public Object next() {
            if (m_triple==null)
                throw new NoSuchElementException();
            Object[] triple=m_triple;
            m_triple=null;
            return triple;
        }
        public void remove()  {
            throw new UnsupportedOperationException();
        }
    }

    /**
     * Checker for the binding patterns of predicates.
     */
    protected static class PredicateBindingPatternsImpl implements PredicateBindingPatterns {
        /**
         * Determines whether given binding pattern is allowed for given predicate.
         *
         * @param predicate                         the predicate
         * @param boundValues                       the bound values
         * @return                                  <code>true</code> if given binding pattern is allowed for given predicate
         */
        public boolean isBindingPatternAllowed(Predicate predicate,boolean[] boundValues) {
            String predicateFullName=predicate.getFullName();
            if ("IsResource/1".equals(predicateFullName) || "IsLiteral/1".equals(predicateFullName) || "IsVisible/1".equals(predicateFullName))
                return boundValues.length==1 && boundValues[0];
            else
                return true;
        }
    }
}
