package edu.unika.aifb.kaon.apionrdf;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Implementation of the Concept interface that relies on the RDF API as the model.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class ConceptImpl extends AbstractEntityImpl implements Concept {
    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                   OI-model
     * @param resource                  resource
     */
    public ConceptImpl(OIModelImpl oimodel,Resource resource) {
        super(oimodel,resource);
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() {
        return OIModel.LOAD_CONCEPT_ALL;
    }
    /**
     * Tests whether this concept is part of its associated OI-model.
     *
     * @return                          <code>true</code> if concpet is a member in the OI-model
     */
    public boolean isInOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                return model.contains(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceClass);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. Note that this may return <code>false</code> if this
     * entity was declared in an included OI-model.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                return model.thisContains(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceClass);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model where this entity was declared originally.
     *
     * @return                          the OI-model there this entity was declated originally
     */
    public OIModel getSourceOIModel() throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Statement statement=model.getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceClass);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all properties starting from this concept.
     *
     * @return                          set of properties starting from this concept (may be empty)
     */
    public Set getPropertiesFromConcept() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_oimodel.m_resourceDomain,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(m_oimodel.getProperty(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all properties (including inherited ones) starting from this concept.
     *
     * @return                          set of all properties starting from this concept (may be empty)
     */
    public Set getAllPropertiesFromConcept() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getPropertiesFromConcept());
            Iterator superConcepts=getSuperConcepts().iterator();
            while (superConcepts.hasNext()) {
                Concept concept=(Concept)superConcepts.next();
                result.addAll(concept.getAllPropertiesFromConcept());
            }
            return result;
        }
    }
    /**
     * Returns all properties pointing to this concept.
     *
     * @return                          set of properties poining to this concept (may be empty)
     */
    public Set getPropertiesToConcept() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_oimodel.m_resourceRange,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(m_oimodel.getProperty(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all properties (including inherited ones) pointing to this concept.
     *
     * @return                          set of all properties pointing to this concept (may be empty)
     */
    public Set getAllPropertiesToConcept() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getPropertiesToConcept());
            Iterator superConcepts=getSuperConcepts().iterator();
            while (superConcepts.hasNext()) {
                Concept concept=(Concept)superConcepts.next();
                result.addAll(concept.getAllPropertiesToConcept());
            }
            return result;
        }
    }
    /**
     * Returns direct subconcepts of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getSubConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(null,m_oimodel.m_resourceSubClassOf,m_resource);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    set.add(m_oimodel.getConcept(statement.subject()));
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all subconcepts (inherited as well) of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getAllSubConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set subConcepts=getSubConcepts();
            Set result=new HashSet(subConcepts);
            Iterator iterator=subConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSubConcepts());
            }
            return result;
        }
    }
    /**
     * Tests if this concept is a direct subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if given concept is a direct subconcept
     */
    public boolean isDirectSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        synchronized (getLock()) {
            try {
                Resource superConceptResource=getResourceForURI(potentialSuperConcept.getURI());
                return getModel().contains(m_resource,m_oimodel.m_resourceSubClassOf,superConceptResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the OI-model containing the superconcept - subconcept association.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          the OI-model in which the super-subconcept association is declared
     */
    public OIModel getSuperSubConceptOIModel(Concept superConcept) throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Resource superConceptResource=getResourceForURI(superConcept.getURI());
                Statement statement=model.getNodeFactory().createStatement(m_resource,m_oimodel.m_resourceSubClassOf,superConceptResource);
                return getOIModelForStatement(statement);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns <code>true</code> if this concept has been declared a subconcept of given concept in this OI-model.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          <code>true</code> if supplied concept has been declared a subconcept of this concept in this OI-model
     */
    public boolean isSuperSubConceptDeclaredLocally(Concept superConcept) throws KAONException {
        synchronized (getLock()) {
            try {
                Model model=getModel();
                Resource superConceptResource=getResourceForURI(superConcept.getURI());
                return model.thisContains(m_resource,m_oimodel.m_resourceSubClassOf,superConceptResource);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Tests if this concept is a subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if given concept is a subconcept
     */
    public boolean isSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        synchronized (getLock()) {
            // this may be implemented more efficiently...
            return getAllSuperConcepts().contains(potentialSuperConcept);
        }
    }
    /**
     * Returns direct superconcepts of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getSuperConcepts() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                Model result=getModel().find(m_resource,m_oimodel.m_resourceSubClassOf,null);
                Iterator statements=result.iterator();
                while (statements.hasNext()) {
                    Statement statement=(Statement)statements.next();
                    if (statement.object() instanceof Resource)
                        set.add(m_oimodel.getConcept((Resource)statement.object()));
                    else
                        throw new KAONException("Concept can not be a subconcept of a literal value.");
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns all superconcepts (inherited as well) of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getAllSuperConcepts() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set superConcepts=getSuperConcepts();
            Set result=new HashSet(superConcepts);
            Iterator iterator=superConcepts.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                result.addAll(concept.getAllSuperConcepts());
            }
            return result;
        }
    }
    /**
     * Returns the set of direct instances of this concept.
     *
     * @return                          set of instances of this concept
     */
    public Set getInstances() throws KAONException {
        synchronized (getLock()) {
            try {
                Set set=new HashSet();
                String rootURI=KAONVocabularyAdaptor.INSTANCE.getRoot();
                if (m_resource.getURI().equals(rootURI)) {
                    Set subConcepts=getSubConcepts();
                    Iterator iterator=subConcepts.iterator();
                    while (iterator.hasNext()) {
                        Concept subConcept=(Concept)iterator.next();
                        set.addAll(subConcept.getAllInstances());
                    }
                    Iterator concepts=m_oimodel.getConcepts().iterator();
                    while (concepts.hasNext()) {
                        Concept concept=(Concept)concepts.next();
                        if (!concept.getURI().equals(rootURI))
                            set.add(concept.getSpanningInstance());
                    }
                    Iterator properties=m_oimodel.getProperties().iterator();
                    while (properties.hasNext()) {
                        Property property=(Property)properties.next();
                        set.add(property.getSpanningInstance());
                    }
                }
                else {
                    Resource conceptID=getResourceForURI(getURI());
                    Model result=getModel().find(null,m_oimodel.m_resourceInstanceOf,conceptID);
                    Iterator statements=result.iterator();
                    while (statements.hasNext()) {
                        Statement statement=(Statement)statements.next();
                        set.add(m_oimodel.getInstance(statement.subject()));
                    }
                }
                return set;
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
    }
    /**
     * Returns the set of all instances (even of subconcepts) of a given concept in the pool.
     *
     * @return                          set of all instances of this concept
     */
    public Set getAllInstances() throws KAONException {
        synchronized (getLock()) {
            // if there are hierarcy cycles, this method will loop forever
            Set result=new HashSet(getInstances());
            Set subConcepts=getSubConcepts();
            Iterator iterator=subConcepts.iterator();
            while (iterator.hasNext()) {
                Concept subConcept=(Concept)iterator.next();
                result.addAll(subConcept.getAllInstances());
            }
            return result;
        }
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
}
