package edu.unika.aifb.kaon.apionrdf;

import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.ArrayList;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Visitor of named entity pool change events.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class ApplyChangeVisitor implements ChangeVisitor {
    /** OI-model of this visitor. */
    protected OIModelImpl m_oimodel;
    /** Visitor for processing add events. */
    protected EntityVisitor m_addEntityVisitor;
    /** Visitor for processing remove events. */
    protected EntityVisitor m_removeEntityVisitor;

    /**
     * Creates an instance of this class.
     */
    public ApplyChangeVisitor() {
        m_addEntityVisitor=createAddEntityVisitor();
        m_removeEntityVisitor=createRemoveEntityVisitor();
    }
    /**
     * Creates a visitor for processing add events.
     *
     * @return                          entity visitor for processing add events
     */
    protected EntityVisitor createAddEntityVisitor() {
        return new AddEntityVisitor();
    }
    /**
     * Creates a visitor for processing remove events.
     *
     * @return                          entity visitor for processing remove events
     */
    protected EntityVisitor createRemoveEntityVisitor() {
        return new RemoveEntityVisitor();
    }
    /**
     * Returns a resource with given URI. Resource is not added to the model.
     *
     * @param uri                   URI for given resource
     * @return                      resource with given URI
     */
    protected Resource getResourceForURI(String uri) throws KAONException {
        return m_oimodel.getResourceForURI(uri);
    }
    /**
     * Returns a model of the ontology.
     *
     * @return                      model of the ontology
     */
    protected Model getModel() {
        return m_oimodel.getModel();
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            event.getEntity().accept(m_addEntityVisitor);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            event.getEntity().accept(m_removeEntityVisitor);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource superConceptID=getResourceForURI(event.getSuperConcept().getURI());
            Resource subConceptID=getResourceForURI(event.getSubConcept().getURI());
            Statement conceptIsSubClass=model.getNodeFactory().createStatement(subConceptID,m_oimodel.m_resourceSubClassOf,superConceptID);
            model.add(conceptIsSubClass);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource superConceptID=getResourceForURI(event.getSuperConcept().getURI());
            Resource subConceptID=getResourceForURI(event.getSubConcept().getURI());
            Statement conceptIsSubClass=model.getNodeFactory().createStatement(subConceptID,m_oimodel.m_resourceSubClassOf,superConceptID);
            model.remove(conceptIsSubClass);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            Statement conceptIsDomain=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceDomain,conceptID);
            model.add(conceptIsDomain);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            Statement conceptIsDomain=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceDomain,conceptID);
            model.remove(conceptIsDomain);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            Statement conceptIsRange=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceRange,conceptID);
            model.add(conceptIsRange);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            Statement conceptIsRange=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceRange,conceptID);
            model.remove(conceptIsRange);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            Statement literalIsRange=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceRange,m_oimodel.m_resourceLiteral);
            event.setOldIsAttirubte(model.contains(literalIsRange));
            if (event.getIsAttribute())
                model.add(literalIsRange);
            else
                model.remove(literalIsRange);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            int oldCardinality=setCardinality(event.getProperty(),event.getConcept(),event.getCardinality(),m_oimodel.m_resourceMinCardinality,0);
            event.setOldCardinality(oldCardinality);
        }
        finally {
            m_oimodel=null;
        }
    }
    protected int setCardinality(Property property,Concept concept,int cardinality,Resource cardinalityResource,int defaultValue) throws KAONException {
        try {
            int oldCardinality;
            Model model=getModel();
            Resource propertyID=getResourceForURI(property.getURI());
            Statement oldCardinalityStatement=getCardinalityStatement(propertyID,concept.getURI(),cardinalityResource);
            if (oldCardinalityStatement==null)
                oldCardinality=defaultValue;
            else {
                String cardinalityLabel=((Literal)oldCardinalityStatement.object()).getLabel();
                int terminatingBracket=cardinalityLabel.indexOf(']');
                try {
                    String numberString=cardinalityLabel.substring(terminatingBracket+1);
                    oldCardinality=Integer.parseInt(numberString);
                }
                catch (NumberFormatException e) {
                    throw new KAONException("Invalid cardinality label '"+cardinalityLabel+"'.");
                }
                model.remove(oldCardinalityStatement);
            }
            if (cardinality!=defaultValue) {
                String cardinalityLabel="["+concept.getURI()+"]"+String.valueOf(cardinality);
                Literal cardinalityLiteral=model.getNodeFactory().createLiteral(cardinalityLabel);
                Statement cardinalityStatement=model.getNodeFactory().createStatement(propertyID,cardinalityResource,cardinalityLiteral);
                model.add(cardinalityStatement);
            }
            return oldCardinality;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    protected Statement getCardinalityStatement(Resource propertyID,String conceptURI,Resource cardinalityResource) throws ModelException {
        Model result=getModel().find(propertyID,cardinalityResource,null);
        Iterator statements=result.iterator();
        while (statements.hasNext()) {
            Statement statement=(Statement)statements.next();
            Object object=statement.object();
            if (object instanceof Literal) {
                String cardinalityLabel=((Literal)object).getLabel();
                if (cardinalityLabel.startsWith("[")) {
                    int terminatingBracket=cardinalityLabel.indexOf(']');
                    if (terminatingBracket!=-1) {
                        String storedConceptURI=cardinalityLabel.substring(1,terminatingBracket);
                        if (conceptURI.equals(storedConceptURI))
                            return statement;
                    }
                }
            }
        }
        return  null;
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            int oldCardinality=setCardinality(event.getProperty(),event.getConcept(),event.getCardinality(),m_oimodel.m_resourceMaxCardinality,Integer.MAX_VALUE);
            event.setOldCardinality(oldCardinality);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource superPropertyID=getResourceForURI(event.getSuperProperty().getURI());
            Resource subPropertyID=getResourceForURI(event.getSubProperty().getURI());
            Statement propertyIsSubProperty=model.getNodeFactory().createStatement(subPropertyID,m_oimodel.m_resourceSubPropertyOf,superPropertyID);
            model.add(propertyIsSubProperty);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource superPropertyID=getResourceForURI(event.getSuperProperty().getURI());
            Resource subPropertyID=getResourceForURI(event.getSubProperty().getURI());
            Statement propertyIsSubProperty=model.getNodeFactory().createStatement(subPropertyID,m_oimodel.m_resourceSubPropertyOf,superPropertyID);
            model.remove(propertyIsSubProperty);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            if (event.getProperty1().getInverseProperty()!=null)
                throw new KAONException("Property "+event.getProperty1().getURI()+" already has an inverse property.");
            if (event.getProperty2().getInverseProperty()!=null)
                throw new KAONException("Property "+event.getProperty2().getURI()+" already has an inverse property.");
            Model model=getModel();
            Resource property1ID=getResourceForURI(event.getProperty1().getURI());
            Resource property2ID=getResourceForURI(event.getProperty2().getURI());
            Statement property1IsInverseOfProperty2=model.getNodeFactory().createStatement(property1ID,m_oimodel.m_resourceInverse,property2ID);
            model.add(property1IsInverseOfProperty2);
            Statement property2IsInverseOfProperty1=model.getNodeFactory().createStatement(property2ID,m_oimodel.m_resourceInverse,property1ID);
            model.add(property2IsInverseOfProperty1);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            if (event.getProperty1().getInverseProperty()==null)
                throw new KAONException("Property "+event.getProperty1().getURI()+" doesn't have an inverse property.");
            if (event.getProperty2().getInverseProperty()==null)
                throw new KAONException("Property "+event.getProperty2().getURI()+" doesn't have an inverse property.");
            if (!event.getProperty1().getInverseProperty().equals(event.getProperty2()))
                throw new KAONException("Property "+event.getProperty1().getURI()+" is not an inverse of "+event.getProperty2().getURI()+".");
            Model model=getModel();
            Resource property1ID=getResourceForURI(event.getProperty1().getURI());
            Resource property2ID=getResourceForURI(event.getProperty2().getURI());
            Statement property1IsInverseOfProperty2=model.getNodeFactory().createStatement(property1ID,m_oimodel.m_resourceInverse,property2ID);
            model.remove(property1IsInverseOfProperty2);
            Statement property2IsInverseOfProperty1=model.getNodeFactory().createStatement(property2ID,m_oimodel.m_resourceInverse,property1ID);
            model.remove(property2IsInverseOfProperty1);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            boolean oldValue=false;
            Iterator result=model.find(propertyID,m_oimodel.m_resourceSymmetric,null).iterator();
            while (result.hasNext()) {
                Statement statement=(Statement)result.next();
                RDFNode object=statement.object();
                oldValue=oldValue || m_oimodel.m_literalTrue.equals(object);
                model.remove(statement);
            }
            if (event.getSymmetric()) {
                Statement propertyIsSymmetric=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceSymmetric,m_oimodel.m_literalTrue);
                model.add(propertyIsSymmetric);
            }
            event.setOldSymmetric(oldValue);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Model model=getModel();
            Resource propertyID=getResourceForURI(event.getProperty().getURI());
            boolean oldValue=false;
            Iterator result=model.find(propertyID,m_oimodel.m_resourceTransitive,null).iterator();
            while (result.hasNext()) {
                Statement statement=(Statement)result.next();
                RDFNode object=statement.object();
                oldValue=oldValue || m_oimodel.m_literalTrue.equals(object);
                model.remove(statement);
            }
            if (event.getTransitive()) {
                Statement propertyIsTransitive=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceTransitive,m_oimodel.m_literalTrue);
                model.add(propertyIsTransitive);
            }
            event.setOldTransitive(oldValue);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            if (!conceptID.getURI().equals(KAONVocabularyAdaptor.INSTANCE.getRoot())) {
                Model model=getModel();
                Resource instanceID=getResourceForURI(event.getInstance().getURI());
                Statement instanceIsOfClass=model.getNodeFactory().createStatement(instanceID,m_oimodel.m_resourceInstanceOf,conceptID);
                model.add(instanceIsOfClass);
            }
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            Resource conceptID=getResourceForURI(event.getConcept().getURI());
            if (!conceptID.getURI().equals(KAONVocabularyAdaptor.INSTANCE.getRoot())) {
                Model model=getModel();
                Resource instanceID=getResourceForURI(event.getInstance().getURI());
                Statement instanceIsOfClass=model.getNodeFactory().createStatement(instanceID,m_oimodel.m_resourceInstanceOf,conceptID);
                model.remove(instanceIsOfClass);
            }
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            addPropertyInstance(event.getPropertyInstance());
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            removePropertyInstance(event.getPropertyInstance());
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for settomg the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            PropertyInstance propertyInstance=event.getPropertyInstance();
            Set existingInstances=propertyInstance.getSourceInstance().getFromPropertyInstances();
            List toRemove=new ArrayList();
            Iterator iterator=existingInstances.iterator();
            while (iterator.hasNext()) {
                PropertyInstance existing=(PropertyInstance)iterator.next();
                if (existing.getProperty().equals(propertyInstance.getProperty()))
                    toRemove.add(existing);
            }
            PropertyInstance[] removedInstances=new PropertyInstance[toRemove.size()];
            toRemove.toArray(removedInstances);
            for (int i=0;i<removedInstances.length;i++)
                removePropertyInstance(removedInstances[i]);
            addPropertyInstance(propertyInstance);
            event.setRemovedInstances(removedInstances);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            PropertyInstance propertyInstance=event.getPropertyInstance();
            removePropertyInstance(propertyInstance);
            addPropertyInstance(m_oimodel.getPropertyInstance(propertyInstance.getProperty(),propertyInstance.getSourceInstance(),event.getNewTargetValue()));
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Adds a property instance to the model.
     *
     * @param propertyInstance                      property instance
     */
    protected void addPropertyInstance(PropertyInstance propertyInstance) throws KAONException {
        try {
            Model model=getModel();
            Resource sourceInstanceID=getResourceForURI(propertyInstance.getSourceInstance().getURI());
            Resource relationID=getResourceForURI(propertyInstance.getProperty().getURI());
            Statement instanceHasProperty;
            if (propertyInstance.getTargetValue() instanceof Instance) {
                Resource targetInstanceID=getResourceForURI(((Instance)propertyInstance.getTargetValue()).getURI());
                instanceHasProperty=model.getNodeFactory().createStatement(sourceInstanceID,relationID,targetInstanceID);
            }
            else {
                Literal targetLiteralID=model.getNodeFactory().createLiteral(propertyInstance.getTargetValue().toString());
                instanceHasProperty=model.getNodeFactory().createStatement(sourceInstanceID,relationID,targetLiteralID);
            }
            model.add(instanceHasProperty);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Removes a property instance from the model.
     *
     * @param propertyInstance                      property instance
     */
    protected void removePropertyInstance(PropertyInstance propertyInstance) throws KAONException {
        try {
            Model model=getModel();
            Resource sourceInstanceID=getResourceForURI(propertyInstance.getSourceInstance().getURI());
            Resource relationID=getResourceForURI(propertyInstance.getProperty().getURI());
            Statement instanceHasProperty;
            if (propertyInstance.getTargetValue() instanceof Instance) {
                Resource targetInstanceID=getResourceForURI(((Instance)propertyInstance.getTargetValue()).getURI());
                instanceHasProperty=model.getNodeFactory().createStatement(sourceInstanceID,relationID,targetInstanceID);
            }
            else {
                Literal targetLiteralID=model.getNodeFactory().createLiteral(propertyInstance.getTargetValue().toString());
                instanceHasProperty=model.getNodeFactory().createStatement(sourceInstanceID,relationID,targetLiteralID);
            }
            model.remove(instanceHasProperty);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            if (!(event.getIncludedOIModel() instanceof OIModelImpl))
                throw new KAONException("Aggregation of a non-RDF OI-model with an RDF-based OI-model is not supported.");
            OIModel includedOIModel=event.getIncludedOIModel();
            if (includedOIModel.getKAONConnection()!=m_oimodel.getKAONConnection())
                throw new KAONException("Included model must belong to the same KAON connection.");
            if (m_oimodel.getIncludedOIModels().contains(includedOIModel))
                throw new KAONException("OI-model '"+includedOIModel.getLogicalURI()+"' is already included.");
            m_oimodel.addIncludedOIModel(includedOIModel);
        }
        finally {
            m_oimodel=null;
        }
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        try {
            m_oimodel=(OIModelImpl)event.getOIModel();
            if (!(event.getIncludedOIModel() instanceof OIModelImpl))
                throw new KAONException("Aggregation of a non-RDF OI-model with an RDF-based OI-model is not supported.");
            OIModelImpl includedOIModel=(OIModelImpl)event.getIncludedOIModel();
            if (includedOIModel.getKAONConnection()!=m_oimodel.getKAONConnection())
                throw new KAONException("Included model must belong to the same KAON connection.");
            if (!m_oimodel.getIncludedOIModels().contains(includedOIModel))
                throw new KAONException("OI-model '"+includedOIModel.getLogicalURI()+"' is not included.");
            m_oimodel.removeIncludedOIModel(includedOIModel);
        }
        finally {
            m_oimodel=null;
        }
    }

    /**
     * Visitor that adds elements to the ontology.
     */
    protected class AddEntityVisitor implements EntityVisitor {
        /**
         * Visits a concpet.
         */
        public void visit(Concept entity) throws KAONException {
            try {
                Model model=getModel();
                Resource conceptID=getResourceForURI(entity.getURI());
                Statement conceptIsClass=model.getNodeFactory().createStatement(conceptID,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceClass);
                model.add(conceptIsClass);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
        /**
         * Visits a property.
         */
        public void visit(Property entity) throws KAONException {
            try {
                Model model=getModel();
                Resource propertyID=getResourceForURI(entity.getURI());
                Statement propertyIsProperty=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceProperty);
                model.add(propertyIsProperty);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) {
            // for instances there is nothing special to do in RDF...
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) {
            // for lexical entries there is nothing special to do in RDF...
        }
    }

    /**
     * Visitor that removes elements to the ontology.
     */
    protected class RemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concpet.
         */
        public void visit(Concept entity) throws KAONException {
            try {
                Model model=getModel();
                Resource conceptID=getResourceForURI(entity.getURI());
                Statement conceptIsClass=model.getNodeFactory().createStatement(conceptID,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceClass);
                model.remove(conceptIsClass);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
        /**
         * Visits a property.
         */
        public void visit(Property entity) throws KAONException {
            try {
                Model model=getModel();
                Resource propertyID=getResourceForURI(entity.getURI());
                Statement propertyIsProperty=model.getNodeFactory().createStatement(propertyID,m_oimodel.m_resourceInstanceOf,m_oimodel.m_resourceProperty);
                model.remove(propertyIsProperty);
            }
            catch (ModelException e) {
                throw new KAONException("RDF exception",e);
            }
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) {
            // for instances there is nothing special to do in RDF...
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) {
            // for lexical entries there is nothing special to do in RDF...
        }
    }

}
