package edu.unika.aifb.kaon.apionrdf;

import java.util.Set;
import java.util.Iterator;

import edu.unika.aifb.rdf.api.model.*;
import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Abstract entity in the OI-model that is the RDF resource.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public abstract class AbstractEntityImpl implements Entity {
    /** RDF resource. */
    protected Resource m_resource;
    /** OI-model of this entity. */
    protected OIModelImpl m_oimodel;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                   OI-model where this entity lives
     * @param resource                  RDF resource
     */
    public AbstractEntityImpl(OIModelImpl oimodel,Resource resource) {
        m_oimodel=oimodel;
        m_resource=resource;
    }
    /**
     * Erases the cached state of the object. The next time something is requested, it will be fetched from the storage.
     */
    public void unload() {
    }
    /**
     * Returns the RDF resource of this object.
     *
     * @return                          the RDF resource of this object
     */
    public Resource getResource() {
        return m_resource;
    }
    /**
     * Returns the OI-model of this object.
     *
     * @return                          OI-model of this object
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }
    /**
     * Returns the URI of this entity.
     *
     * @return                          URI of this entity
     */
    public String getURI() throws KAONException {
        try {
            return m_resource.getURI();
        }
        catch (ModelException e) {
            throw new KAONException("Error reading URI of a resource",e);
        }
    }
    /**
     * Retruns the version of this entity.
     *
     * @return                          the version of the entity
     */
    public int getVersion() throws KAONException {
        if (isInOIModel())
            return 1;
        else
            return Integer.MAX_VALUE;
    }
    /**
     * Returns the model of this entity.
     *
     * @return                          model of this entity
     */
    public Model getModel() {
        return m_oimodel.getModel();
    }
    /**
     * Returns a resource with given URI. Resource is not added to the model.
     *
     * @param uri                       URI for given resource
     * @return                          resource with given URI
     */
    protected Resource getResourceForURI(String uri) throws KAONException {
        try {
            return getModel().getNodeFactory().createResource(uri);
        }
        catch (ModelException e) {
            throw new KAONException("Error creating resource with URI '"+uri+"'.",e);
        }
    }
    /**
     * Returns <code>true</code> if resource is in skip set.
     *
     * @param resource                  resource
     * @return                          <code>true</code> if the resource is in the skip set
     */
    protected boolean isInSkipSet(Resource resource) {
        return m_oimodel.getSkipSet().contains(resource);
    }
    /**
     * Returns a spanning instance for this concept.
     *
     * @return                          spanning instance for this concept
     */
    public Instance getSpanningInstance() throws KAONException {
        return m_oimodel.getInstance(m_resource);
    }
    /**
     * Returns all lexical entries associated with this entity.
     *
     * @return                              set of lexical entries associated with this entity
     */
    public Set getLexicalEntries() throws KAONException {
        synchronized (getLock()) {
            Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
            return getSpanningInstance().getToPropertyValues(references);
        }
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI) throws KAONException {
        synchronized (getLock()) {
            Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
            Set set=getSpanningInstance().getToPropertyValues(references);
            Iterator iterator=set.iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
                if (!lexicalEntry.getTypeURI().equals(typeURI))
                    iterator.remove();
            }
            return set;
        }
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI,String languageURI) throws KAONException {
        synchronized (getLock()) {
            Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
            Set set=getSpanningInstance().getToPropertyValues(references);
            Iterator iterator=set.iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
                if (!lexicalEntry.getTypeURI().equals(typeURI) || !lexicalEntry.getInLanguage().equals(languageURI))
                    iterator.remove();
            }
            return set;
        }
    }
    /**
     * Retruns a lexical entry of specified type and in given language associated with this entity. If there are more
     * lexical entries that match the criteria, one is picked and returned. If there are no lexical entries, <code>null</code>
     * is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              lexical entries of given type and language, or <code>null</code> if no entry is found
     */
    public LexicalEntry getLexicalEntry(String typeURI,String languageURI) throws KAONException {
        synchronized (getLock()) {
            Set set=getLexicalEntries(typeURI);
            Iterator iterator=set.iterator();
            while (iterator.hasNext()) {
                LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
                if (languageURI.equals(lexicalEntry.getInLanguage()))
                    return lexicalEntry;
            }
            return null;
        }
    }
    /**
     * Returns the value of the lexical entry of given type in given language with given attribute name. If no appropriate
     * information is found, <code>null</code> is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @param attributeURI                  URI name of the attribute that is requested
     * @return                              value of given attribute
     */
    public String getLexicalAttribute(String typeURI,String languageURI,String attributeURI) throws KAONException {
        synchronized (getLock()) {
            LexicalEntry lexicalEntry=getLexicalEntry(typeURI,languageURI);
            if (lexicalEntry==null)
                return null;
            else {
                Object value=lexicalEntry.getAttribute(attributeURI);
                if (value instanceof String)
                    return (String)value;
                else
                    return null;
            }
        }
    }
    /**
     * Returns the label in given language.
     *
     * @param languageURI                   URI of the language
     * @return                              label in given language
     */
    public String getLabel(String languageURI) throws KAONException {
        return getLexicalAttribute(KAONVocabularyAdaptor.INSTANCE.getKAONLabel(),languageURI,KAONVocabularyAdaptor.INSTANCE.getValue());
    }
    /**
     * Returns <code>true</code> if supplied statement is in the OI-model.
     *
     * @param statement                     the RDF statement
     * @return                              <code>true</code> if the statement is in the RDF model
     */
    protected boolean containsStatement(Statement statement) throws KAONException {
        try {
            Model model=getModel();
            return model.contains(statement);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns the OI-model containing the supplied statement.
     *
     * @param statement                     the RDF statement
     * @return                              the OI-model containing the supplied statement (or <code>null</code> if the OI-model doesn't contain the statement)
     */
    protected OIModel getOIModelForStatement(Statement statement) throws KAONException {
        try {
            Model model=getModel();
            if (model.thisContains(statement))
                return m_oimodel;
            Iterator allModels=model.getAllIncludedModels().iterator();
            while (allModels.hasNext()) {
                Model includedModel=(Model)allModels.next();
                if (includedModel.thisContains(statement))
                    return ((KAONConnectionImpl)m_oimodel.getKAONConnection()).getOIModelForRDFModelMustExist(includedModel);
            }
            return null;
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns <code>true</code> if the statement has been declared locally
     *
     * @param statement                     the RDF statement
     * @return                              <code>true</code> if the statement has been declared locally
     */
    protected boolean containsStatementLocally(Statement statement) throws KAONException {
        try {
            Model model=getModel();
            return model.thisContains(statement);
        }
        catch (ModelException e) {
            throw new KAONException("RDF exception",e);
        }
    }
    /**
     * Returns the lock object for this entity.
     *
     * @return                              the lock object for the entity
     */
    protected Object getLock() {
        return m_oimodel.m_kaonConnection;
    }
}
