package com.ibm.able.examples;

//====================================================================
//
// Licensed Materials - Property of IBM
//
// "Restricted Materials of IBM"
//
// Product: com.ibm.able.examples
//
// (C) Copyright IBM Corp. 2000 All Rights Reserved.
//
// US Government Users Restricted Rights - Use, duplication or
// disclosure restricted by GSA ADP Schedule Contract with
// IBM Corp.
//
//
//                             DISCLAIMER
//                             ----------
//
// This material contains programming source code for your consideration.
// These examples have not been thoroughly tested under all conditions.
// IBM, therefore, cannot guarantee or imply reliability, serviceability,
// performance or function of these programs.  All programs contained
// herein are provided to you "AS IS".  THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY
// DISCLAIMED.
//
//====================================================================

import com.ibm.able.Able;
import com.ibm.able.AbleDefaultAgent;
import com.ibm.able.AbleObject;

/**
 * The AbleTraceLogDemo class illustrates starting, changing levels, and stopping
 * tracing support using both the ABLE package {@link Able#TraceLog} as a global logger
 * and the use of bean-level tracing which enables individual beans to trace statements
 * at a different level than the global trace logger. 
 * <p>
 * The demo begins by ensuring that tracing is off which is the preferences default.
 * It then sets the console trace level, and calls a method which is instrumented to display
 * trace output at a medium level.  After changing the level, it repeats the call to the 
 * method. 
 * <p>
 * Next it shows how text data and exception information is passed to the trace logger.
 * <p>
 * Finally it constructs a bean and configures it for bean level tracing.  At varying bean
 * trace level and global trace levels, it displays trace outputs from beans using bean 
 * tracing and global tracing. 
 * <p>
 * The global message logger {@link Able#MessageLog} works similarly, although
 * message logging is not supported at the bean level.
 * @version  $Revision: 1.5 $, $Date: 2001/09/26 20:17:01 $
 * 
 */
public class AbleTraceLogDemo {

   public static void main(String[] args) {
      try {
         AbleDefaultAgent agent = new AbleDefaultAgent("AbleLogAndTraceAgent");

         // this is how the saved log and trace settings are started from 
         // a static initializer in the Able class.
         // an application does not need to do this.  
         //Able.startMessageAndTraceLogging();

         // if you had no preferences saved, the only logging and trace functions
         // would be logging error level messages on the console
         // let's just be sure your preferences were as expected
         Able.setDefaultMessageAndTraceLogging();
         // set levels to those defaults, no trace logging

         // we depend on the init method displaying entry trace output at medium level
         agent.init(); // no output - no trace logging in default

         Able.setTraceConsoleHandlerLevel(Able.TRC_HIGH);
         //starts trace console handler thread
         agent.init(); // call something, at trace high, console output appears
         
         // show how to send text and exception data on the fly
         agent.getTraceLogger().text(Able.TRC_MEDIUM,"Medium trace also appears at High");
         Able.TraceLog.exception(Able.TRC_LOW,"com.ibm.able.examples.AbleTraceLogDemo","main",new Exception("Oops!"));

         Able.setTraceConsoleHandlerLevel(Able.TRC_LOW);
         // change the level, thread running
         agent.init(); // no trace output from init at low level

         AbleObject bean = new AbleObject("AbleLogAndTraceBean");
         bean.init(); // no trace output from init at low level

         Able.setTraceConsoleHandlerLevel(Able.TRC_MEDIUM);
         // change the level, thread running
         bean.init(); // get trace output from bean at medium level

         // tracing at the bean level allows different levels to be used for different objects
         Able.startBeanTraceLogging(bean);
         // cause a new trace logger to be created, start new handler thread because global has console handler
         Able.setTraceConsoleHandlerLevel(Able.TRC_LOW);
         bean.init();
         // init entry - level of bean console handler was copied from global
         agent.init(); // no output
         Able.setBeanTraceConsoleHandlerLevel(bean, Able.TRC_LOW); // 
         bean.init(); // no output
         agent.init(); // no output
         Able.setBeanTraceConsoleHandlerLevel(bean, Able.TRC_MEDIUM);
         bean.init(); // output because bean is at medium
         agent.init(); // no output because global is at low
         Able.setTraceConsoleHandlerLevel(Able.TRC_MEDIUM);
         Able.setBeanTraceConsoleHandlerLevel(bean, Able.TRC_LOW);
         bean.init();
         // no output from bean handler, bean now at low, but do get output from global handler
         agent.init(); // output because global is now medium
         Able.stopBeanTraceLogging(bean);
         // end the bean's console handler thread
         bean.init(); // bean is now global, which is medium so output shown
         agent.init(); // output because global is now medium
         Able.setTraceConsoleHandlerLevel(Able.TRC_LOW);
         bean.init(); // no output
         agent.init(); // no output
         Able.setTraceConsoleHandlerLevel(Able.TRC_NONE);  // end the console handler thread
         // or
         //Able.stopTraceLogging();  // ends all handlers

      } catch (Exception exc) {
         System.out.println(exc.getMessage());
         exc.printStackTrace();
      }
   }
}
