unit HSMMain;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, Math, TeEngine, Series, ExtCtrls, TeeProcs, Chart, StdCtrls,
  ImgList, ComCtrls, ToolWin;

const
  NUM_MNEMONICS = 30;              // number of mnemonics to create
  MNEMONICS_PER_SYSTEM = 5;        // number of mnemonics per system
  MAX_SERIES_COUNT = 50;           // maximum points to graph per mnemonic
  GENERATE_CHARTS = TRUE ;         // generate and show graphs for mnemonics
  DISPLAY_ALERTS = TRUE ;          // Display Mnemonic, System and Satellite Status
  INTERVAL_TIME: Integer = 500;    // how much time passes per time step (ms)
  SIM_SPEED: Integer = 100;        // how fast should sim run (ms)
  START_WITH_RUNNING = FALSE;      // start application with simulation running?
  // the next constants direct which expert system to load
  EXPERT_SYSTEM = 'expert system'; // use this constant to force app to ask user for name
//  EXPERT_SYSTEM = 'hsm - all systems.ie';
//  EXPERT_SYSTEM = 'hsm - system basic, no science.ie';
//  EXPERT_SYSTEM = 'hsm - satellite basic, no science.ie';
//  EXPERT_SYSTEM = 'hsm - satellite, no science.ie';

{ This types help define the structure of our example spacecraft }
type
  TSystems = ( Communications, Power, Control, Scientific );
  TSystemNames = Array[TSystems] of String;
var
  Systems: TSystemNames = ('Communications', 'Power', 'Control', 'Scientific');
  SystemMnemPrefix: TSystemNames = ('Comm', 'Pwr', 'Ctrl', 'Sci');
  DEBUG_MONITOR_ON: Boolean = FALSE;        // Bring up Console after each iteration,
                                   // but before running IE
  MONITOR_ON: Boolean = TRUE ;              // turn on expert system

type
  TFormula = (fmSin, fmCos, fmArcCos, fmArcSin, fmTan, fmCot, fmSec, fmCsc);

  TfrmHSM = class;
  TMnemonics = class;
  TMnemonic = class(TCollectionItem)
  { Purpose: To encapsulate one mnemonic value that we have to monitor.  It contains
    information to generate next value }
  private
    { Private declarations }
    FName: String;
    FFormula: TFormula;
    FRawValue: Double;
    FValue: Double;
    FModifier: Double;
    FRedLow: Double;
    FYellowLow: Double;
    FRedHigh: Double;
    FYellowHigh: Double;
    FIncrement: Double;
    FTag: Integer;
    FSystem: String;
    FSubSystem: String;
    function GetCollection: TMnemonics;
    procedure SetCollection(const Value: TMnemonics); reintroduce;
    procedure SetName(const Value: String);
    procedure SetFormula(const Value: TFormula);
    procedure SetRawValue(const Value: Double);
    procedure SetValue(const Value: Double);
    procedure SetModifier(const Value: Double);
    procedure SetRedHigh(const Value: Double);
    procedure SetRedLow(const Value: Double);
    procedure SetYellowHigh(const Value: Double);
    procedure SetYellowLow(const Value: Double);
    procedure SetIncrement(const Value: Double);
    procedure SetSystem(const Value: String);
    procedure SetSubSystem(const Value: String);
  protected
    { Protected declarations }
    function GetDisplayName: String; override;
  public
    { Public declarations }
    procedure Assign(Source: TPersistent); override;
    constructor Create(Collection: TCollection); override;
    property Collection: TMnemonics read GetCollection write SetCollection;
    procedure Calculate; virtual;
    procedure Iterate; virtual;
    function IsYellow: Boolean;
    function IsRed: Boolean;
  published
    { Published declarations }
    property Name: String read FName write SetName;
    property Formula: TFormula read FFormula write SetFormula;
    property RawValue: Double read FRawValue write SetRawValue;
    property Value: Double read FValue write SetValue;
    property Increment: Double read FIncrement write SetIncrement;
    property Modifier: Double read FModifier write SetModifier;
    property RedHigh: Double read FRedHigh write SetRedHigh;
    property RedLow: Double read FRedLow write SetRedLow;
    property YellowHigh: Double read FYellowHigh write SetYellowHigh;
    property YellowLow: Double read FYellowLow write SetYellowLow;
    property Tag: Integer read FTag write FTag;
    property System: String read FSystem write SetSystem;
    property SubSystem: String read FSubSystem write SetSubSystem;
  end; { TMnemonic }
  TMnemonicEvent = procedure (Sender: TObject; Item: TMnemonic ) of object;

  TMnemonics = class(TOwnedCollection)
  { Purpose: To encapsulate a set of mnemonics }
  private
    { Private declarations }
    FOnIterateItem: TMnemonicEvent;
    FOnAfterIterate: TNotifyEvent;
    FOnBeforeIterate: TNotifyEvent;
    function GetItem(Index: Integer): TMnemonic;
    procedure SetItem(Index: Integer; const Value: TMnemonic);
  protected
    { Protected declarations }
    procedure ItemIterated(Item: TMnemonic); virtual;
    procedure Notify(Item: TCollectionItem;
      Action: TCollectionNotification); override;
    procedure Update(Item: TCollectionItem); override;
  public
    { Public declarations }
    constructor Create(AOwner: TfrmHSM);
    function Add: TMnemonic;
    function FindItemID(ID: Integer): TMnemonic;
    function Insert(Index: Integer): TMnemonic;
    property Items[Index: Integer]: TMnemonic read GetItem write SetItem; default;
    function Owner: TfrmHSM; reintroduce;
    function CreateMnemonic( Formula: TFormula; RawValue, Increment, Modifier: Double ): TMnemonic; overload; virtual;
    function CreateMnemonic( Formula: TFormula ): TMnemonic; overload;
    procedure CreateMnemonics( Num: Integer ); virtual;
    procedure Iterate; virtual;
  published
    { Published declarations }
    property OnAfterIterate: TNotifyEvent read FOnAfterIterate write FOnAfterIterate;
    property OnBeforeIterate: TNotifyEvent read FOnBeforeIterate write FOnBeforeIterate;
    property OnIterateItem: TMnemonicEvent read FOnIterateItem write FOnIterateItem;
  end; { TMnemonics }

  TfrmHSM = class(TForm)
    GroupBox1: TGroupBox;
    Splitter1: TSplitter;
    ScrollBox1: TScrollBox;
    Chart1: TChart;
    Timer1: TTimer;
    Series2: TLineSeries;
    lbMnemonics: TListBox;
    Splitter2: TSplitter;
    Label1: TLabel;
    Image1: TImage;
    sbSystems: TScrollBox;
    gbRunParam: TGroupBox;
    GroupBox2: TGroupBox;
    Splitter3: TSplitter;
    Panel1: TPanel;
    Splitter4: TSplitter;
    ToolBar1: TToolBar;
    tbStep: TToolButton;
    ImageList1: TImageList;
    tbPause: TToolButton;
    tbPlay: TToolButton;
    GroupBox3: TGroupBox;
    Splitter5: TSplitter;
    Label2: TLabel;
    lbSciMnems: TListBox;
    tbSimSpeed: TTrackBar;
    Button1: TButton;
    pcSatViews: TPageControl;
    tsImage: TTabSheet;
    TabSheet2: TTabSheet;
    mmLog: TMemo;
    pnlHealth: TPanel;
    cbDebug: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure cbDebugClick(Sender: TObject);
    procedure lbMnemonicsDrawItem(Control: TWinControl; Index: Integer;
      Rect: TRect; State: TOwnerDrawState);
    procedure tbStepClick(Sender: TObject);
    procedure tbPauseClick(Sender: TObject);
    procedure tbPlayClick(Sender: TObject);
    procedure tbSimSpeedChange(Sender: TObject);
    procedure Button1Click(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
    FMnemonics: TMnemonics;
    FTimeStep: TDateTime;
    FSteps: Integer;
    FBaseCaption: String;
    FIsSimRunning: Boolean;
    FProcessingTime: LongWord;
    procedure SetMnemonics(const Value: TMnemonics);
    procedure SetTimeStep(const Value: TDateTime);
    procedure SetSteps(const Value: Integer);
    procedure SetIsSimRunning(const Value: Boolean);
  protected
    { Protected declarations }
    procedure MnemonicNotify(Item: TMnemonic;
      Action: TCollectionNotification); virtual;
    procedure MnemonicUpdate(Item: TMnemonic); virtual;
    procedure ItemIterated( Sender: TObject; Item: TMnemonic );
    procedure Iterating( Sender: TObject ); virtual;
    procedure Iterated( Sender: TObject ); virtual;
  public
    { Public declarations }
    procedure Iterate; virtual;
    procedure Log(const Msg: String); virtual;
    procedure Alert( const Sys: String; Warning: Boolean = False );
    procedure Satellite( Health: Single ); virtual;
    property TimeStep: TDateTime read FTimeStep write SetTimeStep;
    property Mnemonics: TMnemonics read FMnemonics write SetMnemonics;
    property BaseCaption: String read FBaseCaption;
    property Steps: Integer read FSteps write SetSteps;
    property ProcessingTime: LongWord read FProcessingTime write FProcessingTime;
    property IsSimRunning: Boolean read FIsSimRunning write SetIsSimRunning;
  end;

var
  frmHSM: TfrmHSM;

implementation

uses
    TypInfo, DateUtils, HSMDataModule, MMSystem;
    
{$R *.dfm}

{ TMnemonic }

procedure TMnemonic.Assign(Source: TPersistent);
begin
     if Source is TMnemonic then
     begin
          // Copy properties here
          FName := TMnemonic(Source).Name;
          FFormula := TMnemonic(Source).Formula;
          FRawValue := TMnemonic(Source).RawValue;
          FValue := TMnemonic(Source).Value;
          FModifier := TMnemonic(Source).Modifier;
          FRedLow := TMnemonic(Source).RedLow;
          FYellowLow := TMnemonic(Source).YellowLow;
          FRedHigh := TMnemonic(Source).RedHigh;
          FYellowHigh := TMnemonic(Source).YellowHigh;
          Changed(False);
     end
     else
         inherited Assign(Source);
end;

function TMnemonic.GetCollection: TMnemonics;
begin
     result := TMnemonics(inherited Collection);
end;

procedure TMnemonic.SetCollection(const Value: TMnemonics);
begin
     inherited Collection := Value;
end;

function TMnemonic.GetDisplayName: String;
begin
     result := Format('%s (%s): i:%f m:%f', [Name, GetEnumName(TypeInfo(TFormula), Ord(Formula)), Increment, Modifier]);
     if result = '' then
        result := inherited GetDisplayName;
end;

procedure TMnemonic.SetName(const Value: String);
begin
     if Value <> Name then
     begin
          FName := Value;
          Changed(False);
     end;
end;

procedure TMnemonic.SetFormula(const Value: TFormula);
begin
     if Value <> Formula then
     begin
          FFormula := Value;
          Changed(False);
     end;
end;

procedure TMnemonic.SetRawValue(const Value: Double);
begin
     if Value <> RawValue then
     begin
          FRawValue := Value;
          Changed(False);
     end;
end;

procedure TMnemonic.SetValue(const Value: Double);
begin
     FValue := Value;
     Changed(False);
end;

procedure TMnemonic.SetModifier(const Value: Double);
begin
     if Value <> Modifier then
     begin
          FModifier := Value;
          Changed(False);
     end;
end;


procedure TMnemonic.Calculate;
begin
     case Formula of
          fmSin:    Value := Sin(RawValue)*Modifier;
          fmCos:    Value := Cos(RawValue)*Modifier;
          fmArcCos: Value := ArcCos(RawValue)*Modifier;
          fmArcSin: Value := ArcSin(RawValue)*Modifier;
          fmTan: Value := Tan(RawValue)*Modifier;
          fmCot: Value := Cot(RawValue)*Modifier;
          fmSec: Value := Sec(RawValue)*Modifier;
          fmCsc: Value := Csc(RawValue)*Modifier;
     end;
end;

procedure TMnemonic.SetRedHigh(const Value: Double);
begin
     if Value <> RedHigh then
     begin
          FRedHigh := Value;
          if RedHigh < RedLow then
             FRedLow := RedHigh;
          Changed(False);
     end;
end;

procedure TMnemonic.SetRedLow(const Value: Double);
begin
     if Value <> RedLow then
     begin
          FRedLow := Value;
          if RedHigh < RedLow then
             FRedHigh := RedLow;
          Changed(False);
     end;
end;

procedure TMnemonic.SetYellowHigh(const Value: Double);
begin
     if Value <> YellowHigh then
     begin
          FYellowHigh := Value;
          if YellowHigh < YellowLow then
             FYellowLow := YellowHigh;
          Changed(False);
     end;
end;

procedure TMnemonic.SetYellowLow(const Value: Double);
begin
     if Value <> YellowLow then
     begin
          FYellowLow := Value;
          if YellowHigh < YellowLow then
             FYellowHigh := YellowLow;
          Changed(False);
     end;
end;

procedure TMnemonic.Iterate;
begin
     FRawValue := RawValue + Increment;
     if Formula in [fmArcSin, fmArcCos] then
     begin
          if RawValue < -1 then
             FRawValue := 2 + RawValue
          else if RawValue > 1 then
               FRawValue := -2 + RawValue;
     end;
     Calculate;
     if Collection <> nil then
        Collection.ItemIterated(Self);
end;

procedure TMnemonic.SetIncrement(const Value: Double);
begin
     if Value <> Increment then
     begin
          FIncrement := Value;
          Changed(False);
     end;
end;

function TMnemonic.IsRed: Boolean;
begin
     result := (Value > RedHigh) or (Value < RedLow);
end;

function TMnemonic.IsYellow: Boolean;
begin
     result := (Value > YellowHigh) or (Value < YellowLow);
end;

procedure TMnemonic.SetSystem(const Value: String);
begin
     if Value <> System then
     begin
          FSystem := Value;
          Changed(False);
     end;
end;

procedure TMnemonic.SetSubSystem(const Value: String);
begin
  FSubSystem := Value;
end;

constructor TMnemonic.Create(Collection: TCollection);
begin
     inherited Create(Collection);
     FSystem := 'unknown';
     FSubSystem := 'unknown';
end;

{ TMnemonics }

function TMnemonics.Add: TMnemonic;
begin
     result := TMnemonic(inherited Add);
end;

function TMnemonics.FindItemID(ID: Integer): TMnemonic;
begin
     result := TMnemonic(inherited FindItemID(ID));
end;

function TMnemonics.GetItem(Index: Integer): TMnemonic;
begin
     result := TMnemonic(inherited Items[Index]);
end;

function TMnemonics.Owner: TfrmHSM;
var
   AOwner: TPersistent;
begin
     AOwner := inherited Owner;
     if AOwner is TfrmHSM then
        result := TfrmHSM(AOwner)
     else
         result := nil;
end;

function TMnemonics.Insert(Index: Integer): TMnemonic;
begin
     result := TMnemonic(inherited Insert(Index));
end;

procedure TMnemonics.SetItem(Index: Integer; const Value: TMnemonic);
begin
     inherited Items[Index] := Value;
end;

constructor TMnemonics.Create(AOwner: TfrmHSM);
begin
     inherited Create(AOwner, TMnemonic);
end;

procedure TMnemonics.Iterate;
var
   i: Integer;
begin
     if Assigned(FOnBeforeIterate) then FOnBeforeIterate(Self);
     for i := 0 to Count - 1 do
         Items[i].Iterate;
     if Assigned(FOnAfterIterate) then FOnAfterIterate(Self);
end;

procedure TMnemonics.CreateMnemonics(Num: Integer);
begin
     // create some mnemonics
     BeginUpdate;
     try
        while Num > 0 do
        begin
             CreateMnemonic(TFormula(Random(Ord(High(TFormula))+1)));
             Dec(Num);
        end;
     finally
        EndUpdate;
     end;
end;

procedure TMnemonics.Notify(Item: TCollectionItem;
  Action: TCollectionNotification);
begin
     inherited Notify(Item, Action);
     if Owner <> nil then
        Owner.MnemonicNotify(TMnemonic(Item), Action);
end;

procedure TMnemonics.Update(Item: TCollectionItem);
begin
     inherited Update(Item);
     if Owner <> nil then
        Owner.MnemonicUpdate(TMnemonic(Item));
end;

function TMnemonics.CreateMnemonic(Formula: TFormula): TMnemonic;
var
   R, I, M: Double;
begin
     R := Random(361)/360-1;// -1 to 1
     I := Random(101)/100 - 0.5; // -0.5 to 0.5
     M := Random(201)-100; // -100 to 100
     case Formula of
          fmCot, fmTan:
          begin
               I := I * 0.1;
               M := M * 0.1;
          end;
     end;
     result := CreateMnemonic(Formula, R, I, M);
end;

function TMnemonics.CreateMnemonic(Formula: TFormula; RawValue, Increment,
  Modifier: Double): TMnemonic;
begin
     result := Add;
     result.Name := 'Mnem'+IntToStr(Count);
     result.RawValue := RawValue;
     result.Increment := Increment;
     if result.Increment = 0 then
        result.Increment := 0.05;
     result.Modifier := Modifier;
     case Formula of
          fmCsc, fmSec:
          begin
               result.YellowHigh := abs(result.Modifier)*1.5;
               result.YellowLow := -result.YellowHigh;
               result.RedHigh := abs(result.Modifier)*3;
               result.RedLow := -result.RedHigh;
          end;
          fmArcCos:
          begin
               if Sign(Result.Modifier) = PositiveValue then
               begin
                    result.YellowHigh := abs(result.Modifier)*2;
                    result.YellowLow := abs(result.Modifier);
                    result.RedHigh := abs(result.Modifier)*3;//2.25;
                    result.RedLow := abs(result.Modifier)*0.5;//0.75;
               end
               else
               begin
                    result.YellowLow := (result.Modifier)*2;
                    result.YellowHigh := (result.Modifier);
                    result.RedLow := (result.Modifier)*3;//2.25;
                    result.RedHigh := (result.Modifier)*0.5;//0.75;
               end;
          end;
     else
         result.YellowHigh := abs(result.Modifier);//*0.90; // extreme 5% range
         result.YellowLow := -result.YellowHigh;
         result.RedHigh := abs(result.Modifier)*2; // extreme 2% range
         result.RedLow := -result.RedHigh;
     end;
     result.Formula := Formula;
end;

procedure TMnemonics.ItemIterated(Item: TMnemonic);
begin
     if Assigned(FOnIterateItem) then FOnIterateItem(Self, Item);
end;

{ TfrmHSM }

procedure TfrmHSM.SetMnemonics(const Value: TMnemonics);
begin
  FMnemonics.Assign( Value );
end;

procedure TfrmHSM.FormCreate(Sender: TObject);
var
   i: Integer;
   s: TSystems;
begin
     cbDebug.Checked := DEBUG_MONITOR_ON;
     LongTimeFormat := 'h:mm:ss.zzz AMPM';
     pcSatViews.ActivePageIndex := 0;
     FBaseCaption := Caption;
     FTimeStep := Now;
     tbSimSpeed.Position := SIM_SPEED;
     Timer1.Interval := SIM_SPEED;
     FMnemonics := TMnemonics.Create(Self);
     FMnemonics.OnBeforeIterate := Iterating;
     FMnemonics.OnAfterIterate := Iterated;
     FMnemonics.OnIterateItem := ItemIterated;
     RandSeed := 7890;
     Mnemonics.BeginUpdate;
     try
       Mnemonics.CreateMnemonic(fmSin, Random, 0.01, -56);
       Mnemonics.CreateMnemonic(fmArcCos, Random, -0.05, 11);
       Mnemonics.CreateMnemonic(fmCsc, Random, 0.01, 24);
       Mnemonics.CreateMnemonics(NUM_MNEMONICS-Mnemonics.Count);

       // Set up Satellite Simulation
       for s := Low(TSystems) to High(TSystems) do
           with TPanel.Create(Self) do
           begin
                Parent := sbSystems;
                Caption := Systems[s];
                Height := 75;
                Align := alTop;
                Hint := Caption + ' system status (yellow=warning, red=critical)';
           end;
       // Now let's set up the mnemonics for Satellite simulation
       for i := 0 to Mnemonics.Count - 1 do
           if (i div MNEMONICS_PER_SYSTEM) > Ord(High(TSystems)) then // scientific
           begin
                Mnemonics[i].System := Systems[High(TSystems)];
                Mnemonics[i].Name := SystemMnemPrefix[High(TSystems)] +
                                     Mnemonics[i].Name;
           end
           else
           begin
                Mnemonics[i].System := Systems[TSystems(i div MNEMONICS_PER_SYSTEM)];
                Mnemonics[i].Name := SystemMnemPrefix[TSystems(i div MNEMONICS_PER_SYSTEM)] +
                                     Mnemonics[i].Name;
           end;
     finally
       Mnemonics.EndUpdate;
     end;
end;

procedure TfrmHSM.FormDestroy(Sender: TObject);
begin
     FMnemonics.Free;
end;

procedure TfrmHSM.MnemonicNotify(Item: TMnemonic;
  Action: TCollectionNotification);
var
   AChart: TChart;
   Series: TLineSeries;
begin
     if not GENERATE_CHARTS then Exit;
     case Action of
          cnAdded:
          begin
               AChart := TChart.Create(Self);
               AChart.Assign(Chart1);
               AChart.Parent := ScrollBox1;
               with Chart1 do
                    AChart.SetBounds(Left, Top, Width, Height);
               AChart.Align := alLeft;
               Series := TLineSeries.Create(Self);
               Series.Assign(Series2);
               Series.ParentChart := AChart;
               Item.Tag := Integer(Series);
               Series := TLineSeries.Create(AChart);
               Series.ColorEachPoint := False;
               Series.SeriesColor := clYellow;
               Series.ParentChart := AChart;
               Series := TLineSeries.Create(AChart);
               Series.ColorEachPoint := False;
               Series.SeriesColor := clYellow;
               Series.ParentChart := AChart;
               Series := TLineSeries.Create(AChart);
               Series.ColorEachPoint := False;
               Series.SeriesColor := clRed;
               Series.ParentChart := AChart;
               Series := TLineSeries.Create(AChart);
               Series.ColorEachPoint := False;
               Series.SeriesColor := clRed;
               Series.ParentChart := AChart;
          end;
          cnExtracting:
          begin
               Series := TLineSeries(Item.Tag);
               Series.ParentChart.Free;
          end;
     end;
end;

procedure TfrmHSM.MnemonicUpdate(Item: TMnemonic);
var
   i: Integer;
   S: TLineSeries;
begin
     if Item <> nil then
     begin
          lbMnemonics.Invalidate;
          if DISPLAY_ALERTS then
          begin
               i := lbSciMnems.Items.IndexOfObject(Item);
               if (Item.System = Systems[Scientific]) and (i = -1) then
                  lbSciMnems.Items.AddObject( Mnemonics[i].DisplayName, Mnemonics[i] )
               else if (Item.System <> Systems[Scientific]) and (i <> -1) then
                    lbSciMnems.Items.Delete(i);
          end;
          lbSciMnems.Invalidate;
     end
     else // All mnemonics
     begin
          lbMnemonics.Items.BeginUpdate;
          try
             lbMnemonics.Items.Clear;
             for i := 0 to Mnemonics.Count - 1 do
             begin
                  if GENERATE_CHARTS then
                  begin
                       S := TLineSeries( Mnemonics[i].Tag );
                       S.Title := Mnemonics[i].DisplayName;
                       (S.ParentChart as TChart).Title.Text.Text := Mnemonics[i].DisplayName;
                       (S.ParentChart as TChart).Hint := 'Shows last 100 data points for ' + Mnemonics[i].DisplayName + #13#10 +
                                      'Yellow and red lines represent yellow (warning) and red (critical) limits';

                  end;
                  lbMnemonics.Items.AddObject( Mnemonics[i].DisplayName, Mnemonics[i] );
                  if Mnemonics[i].System = Systems[Scientific] then
                     lbSciMnems.Items.AddObject( Mnemonics[i].DisplayName, Mnemonics[i] );
             end;
          finally
             lbMnemonics.Items.EndUpdate;
          end;
     end;
end;

procedure TfrmHSM.Timer1Timer(Sender: TObject);
var
   IsEnabled: Boolean;
begin
     with Sender as TTimer do
     begin
          IsEnabled := IsSimRunning;
          Enabled := False;
          try
             Iterate;
          finally
             Enabled := IsEnabled;
          end;
     end;
end;

procedure TfrmHSM.cbDebugClick(Sender: TObject);
begin
     DEBUG_MONITOR_ON := (Sender as TCheckBox).Checked;
end;

procedure TfrmHSM.SetTimeStep(const Value: TDateTime);
begin
  FTimeStep := Value;
end;

procedure TfrmHSM.lbMnemonicsDrawItem(Control: TWinControl; Index: Integer;
  Rect: TRect; State: TOwnerDrawState);
var
  Flags: Longint;
  Data: String;
begin
     with Control as TListBox do
     begin
          Canvas.FillRect(Rect);
          if Index < Count then
          begin
               Flags := DrawTextBiDiModeFlags(DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
               if not UseRightToLeftAlignment then
                  Inc(Rect.Left, 2)
               else
                   Dec(Rect.Right, 2);
               Data := Items[Index];
               if Items.Objects[Index] is TMnemonic then
               begin
                    if TMnemonic(Items.Objects[Index]).IsRed then
                       Canvas.Brush.Color := clRed
                    else if TMnemonic(Items.Objects[Index]).IsYellow then
                       Canvas.Brush.Color := clYellow;
               end;
               DrawText(Canvas.Handle, PChar(Data), Length(Data), Rect, Flags);
          end;
    end;
end;

procedure TfrmHSM.tbStepClick(Sender: TObject);
begin
     tbPause.Down := True;
     if IsSimRunning then
        IsSimRunning := False
     else
         Iterate;
end;

procedure TfrmHSM.tbPauseClick(Sender: TObject);
begin
     (Sender as TToolButton).Down := True;
     IsSimRunning := False;
end;

procedure TfrmHSM.tbPlayClick(Sender: TObject);
begin
     (Sender as TToolButton).Down := True;
     IsSimRunning := True;
end;

procedure TfrmHSM.tbSimSpeedChange(Sender: TObject);
begin
     Timer1.Interval := (Sender as TTrackBar).Position;
end;

procedure TfrmHSM.Iterated(Sender: TObject);
begin
     Steps := Steps + 1;
     if not MONITOR_ON then Exit;
     // mnemonics have iterated
     DataModule1.Iterated(Sender);
     Application.ProcessMessages;
end;

procedure TfrmHSM.Iterating(Sender: TObject);
var
   i: Integer;
begin
     if not MONITOR_ON then Exit;
     DataModule1.Iterating(Sender);
     for i := 0 to sbSystems.ControlCount - 1 do
         TPanel(sbSystems.Controls[i]).Color := clBtnFace;
end;

procedure TfrmHSM.ItemIterated(Sender: TObject; Item: TMnemonic );
var
   S: TLineSeries;
   ATime: TDateTime;
begin
      if GENERATE_CHARTS then
      begin
          S := TLineSeries(Item.Tag);
          ATime := TimeStep;
          S.AddXY(ATime, Item.Value);
          if Item.IsRed then
             (S.ParentChart as TChart).Title.Color := clRed
          else if Item.IsYellow then
             (S.ParentChart as TChart).Title.Color := clYellow
          else
              (S.ParentChart as TChart).Title.Color := clBtnFace;
          S.ParentChart.Series[1].AddXY(ATime, Item.YellowHigh);
          S.ParentChart.Series[2].AddXY(ATime, Item.YellowLow);
          S.ParentChart.Series[3].AddXY(ATime, Item.RedHigh);
          S.ParentChart.Series[4].AddXY(ATime, Item.RedLow);
          if S.Count > MAX_SERIES_COUNT then
          begin
               S.Delete(0);
               S.ParentChart.Series[1].Delete(0);
               S.ParentChart.Series[2].Delete(0);
               S.ParentChart.Series[3].Delete(0);
               S.ParentChart.Series[4].Delete(0);
          end;
      end;
     if MONITOR_ON then 
        DataModule1.ItemIterated(Sender, Item);
end;

procedure TfrmHSM.SetSteps(const Value: Integer);
begin
     if Value <> Steps then
     begin
          FSteps := Value;
          Caption := BaseCaption + ' ('+IntToStr(Steps)+' points generated, '+
                     IntToStr(trunc(1000 / (MaxValue([ProcessingTime,1]) / Steps))) +
                     ' steps per proc second)';
     end;
end;

procedure TfrmHSM.Button1Click(Sender: TObject);
var
   IsEnabled: Boolean;
begin
     IsEnabled := IsSimRunning;
     Timer1.Enabled := False;
     try
        DataModule1.ConsoleDialog1.Execute;
     finally
        Timer1.Enabled := IsEnabled;
     end;
end;

procedure TfrmHSM.Iterate;
var
   TheTime: LongWord;
begin
     TheTime := TimeGetTime;
     TimeStep := IncMilliSecond(TimeStep,INTERVAL_TIME);
     Mnemonics.Iterate;
     FProcessingTime := ProcessingTime + (TimeGetTime - TheTime);
end;

procedure TfrmHSM.Log(const Msg: String);
begin
     mmLog.Lines.Add(TimeToStr(TimeStep)+ ' ' + Msg);
end;

procedure TfrmHSM.Alert(const Sys: String; Warning: Boolean);
var
   i: Integer;
begin
     if Warning then
        Log('Warning! '+Sys+' degradation detected')
     else
        Log('Alert! '+Sys+' critical');
     if not DISPLAY_ALERTS then Exit;
     for i := 0 to sbSystems.ControlCount - 1 do
         if Sys = (sbSystems.Controls[i] as TPanel).Caption then
         begin
              if Warning then
              begin
                   if TPanel(sbSystems.Controls[i]).Color <> clRed then
                      TPanel(sbSystems.Controls[i]).Color := clYellow;
              end
              else
                 TPanel(sbSystems.Controls[i]).Color := clRed;
         end;
end;

procedure TfrmHSM.SetIsSimRunning(const Value: Boolean);
begin
     FIsSimRunning := Value;
     Timer1.Enabled := Value;
     tbPause.Down := not IsSimRunning;
     tbPlay.Down := IsSimRunning;
end;

procedure TfrmHSM.Satellite(Health: Single);
begin
     if not DISPLAY_ALERTS then Exit;
     // health is a number between 0 and 1, 0 is critical, 1 is healthy
     if Health >= 1 then
        pnlHealth.Color := clBtnFace
     else if Health <= 0 then
          pnlHealth.Color := clRed
     else if Health = 0.5 then
          pnlHealth.Color := clYellow
     else
     begin
          pnlHealth.Color := clAqua;
     {         RD := (Integer(clRed) shl 24) shr 24;
         GD := (Integer(clRed) shl 16) shr 24;
         BD := (Integer(clRed) shl 8) shr 24;
         RD := (RD - (Integer(clGreen) shl 24) shr 24) div 20;
         GD := (GD - (Integer(clGreen) shl 16) shr 24) div 20;
         BD := (BD - (Integer(clGreen) shl 8) shr 24) div 20;
          for i := 1 to trunc( (Progress / (( Max-Min )/Subdivisions)) ) do
          begin
               if (Orientation = trHorizontal) and (BoxOx+BoxW > ClientWidth-3) then
                  FillRect( Rect(trunc(MinValue([ClientWidth-3,BoxOx])), BoxOy, ClientWidth-3, BoxOy+BoxH) )
               else if (Orientation = trVertical) and (BoxOy+BoxH > ClientHeight-3) then
                  FillRect( Rect(BoxOx, trunc(MinValue([ClientHeight-3,BoxOy])), BoxOx+BoxW, ClientHeight-3) )
               else
                  FillRect( Rect(BoxOx, BoxOy, BoxOx+BoxW, BoxOy+BoxH) );
               BoxOx := BoxOx + BoxDeltaX;
               BoxOy := BoxOy + BoxDeltaY;
               if MinColor <> MaxColor then      
               begin
                    RS := ((Integer(Brush.Color) shl 24) shr 24) - RD;
                    GS := (((Integer(Brush.Color) shl 16) shr 24) - GD) shl 8;
                    BS := (((Integer(Brush.Color) shl 8) shr 24) - BD) shl 16;
                    Brush.Color := BS or GS or RS;
               end;
          end;

         RD := trunc(RD*Health);
         GD := trunc(GD*Health);
         BD := trunc(BD*Health);
         pnlHealth.Color := BD or GD or RD;//BS or GS or RS;
         IsSimRunning := False;      }
     end;
end;

procedure TfrmHSM.FormShow(Sender: TObject);
begin
     if Steps = 0 then // first show
     begin
          IsSimRunning := START_WITH_RUNNING;
     end;
end;

end.

