//---------------------------------------------------------------------------

#include <vector>
#include <queue>
#include <algorithm>
#include <functional>
#include <assert.h>

#pragma hdrstop

#include "IntelligenceV01.h"

//---------------------------------------------------------------------------

#pragma package(smart_init)

CIntelligenceV01::CIntelligenceV01( const TIoLimitsContainer& _receptorsLimits,
	const TIoLimitsContainer& _effectorsLimits ) :
	receptorsLimits( _receptorsLimits ), effectorsLimits( _effectorsLimits )
{
}

void CIntelligenceV01::TimeStep( const TIoDataContainer& receptors, TIoDataContainer& effectors )
{
	//	      
	assert( receptors.size() == receptorsLimits.size() );
	TIoDataContainer::const_iterator rData = receptors.begin();
	TIoLimitsContainer::const_iterator rLimits = receptorsLimits.begin();

	for( ; rLimits != receptorsLimits.end(); ++rData, ++rLimits ) {
		assert( *rData >= 0 && *rData < *rLimits );
	}
	//	      
	assert( prevStepEffectors.size() == effectorsLimits.size() );
	TIoDataContainer::const_iterator eData = prevStepEffectors.begin();
	TIoLimitsContainer::const_iterator eLimits = effectorsLimits.begin();

	for( ; eLimits != effectorsLimits.end(); ++eData, ++eLimits ) {
		assert( *eData >= 0 && *eData < *eLimits );
	}
	assert( effectors.size() == effectorsLimits.size() );

	//	 
	TDistributionType adArchive;
	calculateNextStepContext( receptors, adArchive );
	updateStatistics( receptors, adArchive );
	findBestReaction( adArchive, effectors );
}

//	  
void CIntelligenceV01::calculateNextStepContext(
	const TIoDataContainer& receptors, TDistributionType& adArchive ) const
{
	//	   
	assert( worldModel.size() == adArchive.size() );
	TDistributionType::value_type likelihoodSum = 0.;
	TClustersContainer::const_iterator cluster = worldModel.begin();
	TDistributionType::iterator adValue = adArchive.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++adValue ) {
		TDistributionType::value_type likelihood = 0.;

		//	     
		assert( cluster->ContextStatistics.size() >= context.size() );
		TDistributionType::const_iterator cValue = context.begin();
		TDistributionType::const_iterator csValue = cluster->ContextStatistics.begin();

		for( ; cValue != context.end(); ++cValue, ++csValue ) {
			likelihood += ( *cValue ) * ( *csValue + 1. );
		}

		//	      
		likelihood *= cluster->Weight / ( cluster->Weight
			+ static_cast<TDistributionType::value_type>( cluster->ContextStatistics.size() ) );

		//	  
		assert( receptorsLimits.size() == cluster->ReceptorsStatistics.size() );
		TIoDataContainer::const_iterator rValue = receptors.begin();
		TDistributionsContainer::const_iterator rsValue = cluster->ReceptorsStatistics.begin();
		TIoLimitsContainer::const_iterator rLimits = receptorsLimits.begin();

		for( ; rLimits != receptorsLimits.end(); ++rLimits, ++rsValue, ++rValue ) {
			assert( rsValue->size() == *rLimits );
			likelihood *= ( ( *rsValue )[*rValue] + 1. )
				/ ( cluster->Weight + static_cast<TDistributionType::value_type>( *rLimits ) );
		}

		//	  
		assert( effectorsLimits.size() == cluster->EffectorsStatistics.size() );
		TIoDataContainer::const_iterator eValue = prevStepEffectors.begin();
		TDistributionsContainer::const_iterator esValue = cluster->EffectorsStatistics.begin();
		TIoLimitsContainer::const_iterator eLimits = effectorsLimits.begin();

		for( ; eLimits != effectorsLimits.end(); ++eLimits, ++esValue, ++eValue ) {
			assert( esValue->size() == *eLimits );
			likelihood *= ( ( *esValue )[*eValue] + 1. )
				/ ( cluster->Weight + static_cast<TDistributionType::value_type>( *eLimits ) );
		}

		*adValue = likelihood;
		likelihoodSum += likelihood;
	}

	//	    
	assert( likelihoodSum > 0. );
	adValue = adArchive.begin();
	for( ; adValue != adArchive.end(); ++adValue ) {
		*adValue /= likelihoodSum;
	}
}

//	       
void CIntelligenceV01::updateStatistics(
	const TIoDataContainer& receptors, const TDistributionType& adArchive )
{
	assert( worldModel.size() == adArchive.size() );
	TClustersContainer::iterator cluster = worldModel.begin();
	TDistributionType::const_iterator adValue = adArchive.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++adValue ) {
		//	  
		assert( cluster->ContextStatistics.size() >= context.size() );
		TDistributionType::const_iterator cValue = context.begin();
		TDistributionType::iterator csValue = cluster->ContextStatistics.begin();

		for( ; cValue != context.end(); ++cValue, ++csValue ) {
			*csValue += ( *cValue ) * ( *adValue );
		}

		//	  
		assert( receptorsLimits.size() == cluster->ReceptorsStatistics.size() );
		TIoDataContainer::const_iterator rValue = receptors.begin();
		TDistributionsContainer::iterator rsValue = cluster->ReceptorsStatistics.begin();

		for( ; rValue != receptors.end(); ++rValue, rsValue ) {
			( *rsValue )[*rValue] += ( *adValue );
		}

		//    
		assert( effectorsLimits.size() == cluster->EffectorsStatistics.size() );
		TIoDataContainer::const_iterator eValue = prevStepEffectors.begin();
		TDistributionsContainer::iterator esValue = cluster->EffectorsStatistics.begin();

		for( ; eValue < prevStepEffectors.end(); ++eValue, ++esValue ) {
			( *esValue )[*eValue] += ( *adValue );
		}

		cluster->Weight += ( *adValue );
	}
}

//	   
void CIntelligenceV01::findBestReaction( const TDistributionType& adArchive, TIoDataContainer& effectors ) const
{
	//	   
	TDistributionType clustersLikelihood( worldModel.size() );
	prepareClustersLikelihoods( clustersLikelihood, adArchive );

	//	  
	TIoDataContainer firstApproximation( effectorsLimits.size() );
	prepareFirstApproximation( firstApproximation, clustersLikelihood );

	//	  
	TDistributionType preservedLikelihoods( worldModel.size() );
	TDistributionType::value_type bestExpectation = prepareVariant(
		preservedLikelihoods, clustersLikelihood, firstApproximation );

	//	 ,    
	TVariantQueuesContainer queues( effectorsLimits.size(), TVariantQueuesContainer::value_type() );
	TTestedStatesContainer testedStates;
	testedStates.push_back( CEffectorsTestedState( firstApproximation, effectorsLimits ) );
	int bestState = 0;

	for( int i = 0; i < 100; i++ ) {
		effectorVariantsGenerator( queues, preservedLikelihoods, testedStates );

		//	  
		TVariantQueuesContainer::iterator eQueue = queues.begin();
		TVariantQueuesContainer::iterator bestQueue;
		TDistributionType::value_type newBestExpectation = 0.;

		for( ; eQueue != queues.end(); ++eQueue ) {
			if( !eQueue->empty() && eQueue->top().Likelihood > newBestExpectation ) {
				bestQueue = eQueue;
				newBestExpectation = eQueue->top().Likelihood;
			}
		}

		//	   
		if( newBestExpectation > 0. ) {
			CEffectorVariant newVariant = bestQueue->top();
			testedStates.push_back( CEffectorsTestedState( testedStates[newVariant.TestedStateIndex],
				bestQueue - queues.begin(), newVariant.Variant ) );

			newBestExpectation = prepareVariant(
				preservedLikelihoods, clustersLikelihood, testedStates.back().Effectors() );

			if( newBestExpectation > bestExpectation ) {
				bestExpectation = newBestExpectation;
				bestState = testedStates.size() - 1;
			}
		} else {
			break;
		}
	}

	effectors = testedStates[bestState].Effectors();
}

//	   
void CIntelligenceV01::prepareClustersLikelihoods( TDistributionType& clustersLikelihood,
	const TDistributionType& adArchive ) const
{
	assert( clustersLikelihood.empty() );
	TClustersContainer::const_iterator cluster = worldModel.begin();
	TDistributionType::const_iterator adValue = adArchive.begin();
	TDistributionType::const_iterator lValue = adArchive.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++adValue ) {
		TDistributionType::value_type likelihood = 0;

		//	     
		assert( cluster->ContextStatistics.size() >= context.size() );
		TDistributionType::const_iterator cValue = context.begin();
		TDistributionType::const_iterator csValue = cluster->ContextStatistics.begin();

		for( ; cValue != context.end(); ++cValue, ++csValue ) {
			likelihood += ( *cValue ) * ( *csValue + 1. );
		}

		//	      
		clustersLikelihood.push_back( likelihood * cluster->Weight / ( cluster->Weight
			+ static_cast<TDistributionType::value_type>( cluster->ContextStatistics.size() ) ) );
	}
}

//	  
void CIntelligenceV01::prepareFirstApproximation( TIoDataContainer& firstApproximation,
	const TDistributionType& clustersLikelihood ) const
{
	assert( firstApproximation.empty() );
	//	   
	TDistributionsContainer statistics( effectorsLimits.size() );
	TIoLimitsContainer::const_iterator eLimits = effectorsLimits.begin();

	for( ; eLimits != effectorsLimits.end(); ++eLimits ) {
		statistics.push_back( TDistributionType( *eLimits, 0. ) );
	}

	//	 
	TDistributionType::const_iterator clValue = clustersLikelihood.begin();
	TClustersContainer::const_iterator cluster = worldModel.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++clValue ) {
		//	    
		assert( cluster->EffectorsStatistics.size() == effectorsLimits.size() );
		TDistributionsContainer::const_iterator esValue = cluster->EffectorsStatistics.begin();
		TDistributionsContainer::iterator insValue = statistics.begin();

		for( ; insValue != statistics.end(); ++insValue, ++esValue ) {
			//	    
			assert( esValue->size() == insValue->size() );
			TDistributionType::iterator inData = insValue->begin();
			TDistributionType::const_iterator data = esValue->begin();

			for( ; inData != insValue->end(); ++inData, ++data ) {
				*inData += ( *data ) * ( *clValue );
			}
		}
	}

	//	  
	TDistributionsContainer::const_iterator sValue = statistics.begin();

	for( ; sValue != statistics.end(); ++sValue ) {
		firstApproximation.push_back( static_cast<TIoDataContainer::value_type>(
			std::max_element<>( sValue->begin(), sValue->end() ) - sValue->begin() ) );
	}
}

//	  
CIntelligenceV01::TIoDataContainer::value_type CIntelligenceV01::prepareVariant(
	TDistributionType& preservedLikelihoods,
	const TDistributionType& clustersLikelihood, const TIoDataContainer& firstApproximation ) const
{
	assert( preservedLikelihoods.size() == worldModel.size() );
	TDistributionType::value_type bestExpectation = 0;
	TDistributionType::value_type likelihoodSum = 0.;

	TDistributionType::const_iterator clValue = clustersLikelihood.begin();
	TClustersContainer::const_iterator cluster = worldModel.begin();
	TDistributionType::iterator plValue = preservedLikelihoods.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++clValue, ++plValue ) {
		TDistributionType::value_type likelihood = *clValue;

		//	  
		assert( effectorsLimits.size() == cluster->EffectorsStatistics.size() );
		TIoDataContainer::const_iterator eValue = firstApproximation.begin();
		TDistributionsContainer::const_iterator esValue = cluster->EffectorsStatistics.begin();
		TIoLimitsContainer::const_iterator eLimits = effectorsLimits.begin();

		for( ; eLimits != effectorsLimits.end(); ++eLimits, ++esValue, ++eValue ) {
			assert( esValue->size() == *eLimits );
			likelihood *= ( ( *esValue )[*eValue] + 1. )
				/ ( cluster->Weight + static_cast<TDistributionType::value_type>( *eLimits ) );
		}

		*plValue = likelihood;
		bestExpectation += likelihood * ( cluster->Expectation );
		likelihoodSum += likelihood;
	}

	assert( likelihoodSum > 0. );
	return bestExpectation / likelihoodSum;
}

//	  
void CIntelligenceV01::effectorVariantsGenerator( TVariantQueuesContainer& queues,
   const TDistributionType& preservedLikelihoods, const TTestedStatesContainer& lastState ) const
{
	//	    
	TDistributionsContainer otherExpectations( effectorsLimits.size() );
	TDistributionsContainer otherLikelihoodSums( effectorsLimits.size() );
	TIoLimitsContainer::const_iterator eLimits = effectorsLimits.begin();

	for( ; eLimits != effectorsLimits.end(); ++eLimits ) {
		otherExpectations.push_back( TDistributionType( *eLimits, 0. ) );
		otherLikelihoodSums.push_back( TDistributionType( *eLimits, 0. ) );
	}

	//	   
	TClustersContainer::const_iterator cluster = worldModel.begin();
	TDistributionType::const_iterator plValue = preservedLikelihoods.begin();

	for( ; cluster != worldModel.end(); ++cluster, ++plValue ) {
		TDistributionsContainer::const_iterator esValue = cluster->EffectorsStatistics.begin();
		TVariantsContainer::const_iterator eVariant = lastState.back().TestedVariants().begin();
		TIoDataContainer::const_iterator eData = lastState.back().Effectors().begin();
		TDistributionsContainer::iterator oeValue = otherExpectations.begin();
		TDistributionsContainer::iterator olsValue = otherLikelihoodSums.begin();

		//	  
		for( ; olsValue != otherLikelihoodSums.end(); ++esValue, ++eVariant, ++eData, ++oeValue, ++olsValue ) {
			TDistributionType::value_type workLikelihood = *plValue / ( ( *esValue )[*eData] + 1. );
			TVariantsContainer::value_type::const_iterator curVariant = eVariant->begin();
			TDistributionType::const_iterator curEsValue = esValue->begin();
			TDistributionType::iterator curOeValue = oeValue->begin();
			TDistributionType::iterator curOlsValue = olsValue->begin();

			//	   
			for( ; curVariant != eVariant->end(); ++curVariant, ++curEsValue, ++curOeValue, ++curOlsValue ) {
				if( !*curVariant ) {
					TDistributionType::value_type likelihood = *curEsValue * workLikelihood;
					*curOeValue += likelihood * cluster->Expectation;
					*curOlsValue += likelihood;
				}
			}
		}
	}

	//	  
	TVariantQueuesContainer::iterator eQueue = queues.begin();
	TDistributionsContainer::const_iterator oeValue = otherExpectations.begin();
	TDistributionsContainer::const_iterator olsValue = otherLikelihoodSums.begin();

	for( ; eQueue != queues.end(); ++eQueue, ++oeValue, ++olsValue ) {
		for( TIoDataContainer::value_type i = 0; i < oeValue->size(); i++ ) {
			if( ( *olsValue )[i] > 0. ) {
				eQueue->push( CEffectorVariant( ( *oeValue )[i] / ( *olsValue )[i], lastState.size() - 1, i ) );
			}
		}
	}
}

//	    
CIntelligenceV01::CEffectorsTestedState::CEffectorsTestedState(
	const TIoDataContainer& _effectors, const TIoLimitsContainer& limits ) :
	testedVariants( limits.size() ), effectors( _effectors )
{
	TIoLimitsContainer::const_iterator eLimits = limits.begin();
	TIoDataContainer::const_iterator eData = effectors.begin();

	for( ; eLimits != limits.end(); ++eLimits ) {
		testedVariants.push_back( TVariantsContainer::value_type( *eLimits, false ) );
		testedVariants.back()[*eLimits] = true;
	}
}

//	  
CIntelligenceV01::CEffectorsTestedState::CEffectorsTestedState( CEffectorsTestedState& baseState,
	int effectorIndex, TIoDataContainer::value_type newVariant ) :
	testedVariants( baseState.testedVariants ), effectors( baseState.effectors )
{
	assert( !testedVariants[effectorIndex][newVariant] );
	effectors[effectorIndex] = newVariant;
	testedVariants[effectorIndex].assign( testedVariants[effectorIndex].size(), true );
	baseState.testedVariants[effectorIndex][newVariant] = true;
}

