package org.jamocha.xml;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.jamocha.service.ClipsInitialData;
import org.jamocha.service.ClipsRuleset;
import org.jamocha.service.FunctionPackage;
import org.jamocha.service.ObjectModel;
import org.jamocha.service.ServiceConfiguration;
import org.jamocha.service.servlet.RuleServiceApplication;
import org.xml.sax.SAXException;

public class ConfigurationUtility {
	
	private Writer writer = null;
	private String LINE_BREAK = System.getProperty("line.separator");
	private String TAB = "  ";
	private String TAGSTART = "<";
	private String TAGEND = ">";
	private int tabCount = 0;
	
	public ConfigurationUtility() {
		super();
	}
	
	public ServiceConfiguration fromXml(String filename) {
		ConfigurationHandler handler = new ConfigurationHandler();
		
		try {
			File file = new File(filename);
			SAXParser parser = SAXParserFactory.newInstance().newSAXParser();
			parser.parse(file, handler);
			ServiceConfiguration config = handler.getServiceConfiguration();
			return config;
		} catch (ParserConfigurationException e) {
			e.printStackTrace();
		} catch (SAXException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return null;
	}
	
	public void toXml(ServiceConfiguration configuration, String file) {
		try {
			this.writer = new FileWriter(file);
			this.writeConfiguration(configuration);
			this.writer.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	public void writeConfiguration(ServiceConfiguration configuration) throws IOException {
		// Rule service configuration
		this.writeRuleServiceConfiguration(configuration);
	}
	
	protected void writeRuleServiceConfiguration(ServiceConfiguration configuration) throws IOException {
		ServiceConfiguration servConfig = configuration;
		this.writeTabs();
		this.writer.write(this.makeTagStart("ruleServiceConfiguration") + LINE_BREAK);
		this.pushTab();
		
		this.writeTabs();
		this.writer.write(this.makeTagStart("applications") + LINE_BREAK);
		this.pushTab();
		for (int i=0; i < servConfig.getApplications().size(); i++) {
			RuleServiceApplication app = (RuleServiceApplication)servConfig.getApplications().get(i);
			this.writeRuleServiceApplication(app);
		}
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("applications") + LINE_BREAK);
		this.writeTabs();
		this.writer.write(this.makeTagStart("serviceName") + servConfig.getServiceName() + this.makeTagEnd("serviceName") + LINE_BREAK);
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("ruleServiceConfiguration") + LINE_BREAK);
	}
	
	protected void writeRuleServiceApplication(RuleServiceApplication ruleServiceApp) throws IOException {
		this.writeTabs();
		this.writer.write(this.makeTagStart(ruleServiceApp.getClass().getName()) + LINE_BREAK);
		
		this.pushTab();
		this.writeTabs();
		this.writer.write(this.makeTagStart("applicationName") + ruleServiceApp.getName() + this.makeTagEnd("applicationName") + LINE_BREAK);
		this.writeTabs();
		this.writer.write(this.makeTagStart("version") + ruleServiceApp.getVersion() + this.makeTagEnd("version") + LINE_BREAK);
		// models
		this.writeTabs();
		this.writer.write(this.makeTagStart("models") + LINE_BREAK);
		this.pushTab();
		if (ruleServiceApp.getModels() != null) {
			for (int i=0; i < ruleServiceApp.getModels().size(); i++) {
				this.writeModels(ruleServiceApp.getModels().get(i));
			}
		}
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("models") + LINE_BREAK);

		// initial Data
		this.writeTabs();
		this.writer.write(this.makeTagStart("initialData") + LINE_BREAK);
		this.pushTab();
		if (ruleServiceApp.getInitialData() != null) {
			for (int i=0; i < ruleServiceApp.getInitialData().size(); i++) {
				this.writeInitialData(ruleServiceApp.getInitialData().get(i));
			}
		}
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("initialData") + LINE_BREAK);
		
		// function group
		this.writeTabs();
		this.writer.write(this.makeTagStart("functionGroups") + LINE_BREAK);
		this.pushTab();
		if (ruleServiceApp.getFunctionGroups() != null) {
			for (int i=0; i < ruleServiceApp.getFunctionGroups().size(); i++) {
				this.writeFunctionGroup(ruleServiceApp.getFunctionGroups().get(i));
			}
		}
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("functionGroups") + LINE_BREAK);
		
		// rulesets
		this.writeTabs();
		this.writer.write(this.makeTagStart("rulesets") + LINE_BREAK);
		this.pushTab();
		if (ruleServiceApp.getRulesets() != null) {
			for (int i=0; i < ruleServiceApp.getRulesets().size(); i++) {
				this.writeRuleset(ruleServiceApp.getRulesets().get(i));
			}
		}
		this.popTab();
		this.writeTabs();
		this.writer.write(this.makeTagEnd("rulesets") + LINE_BREAK);
		
		// remaining fields
		this.writeTabs();
		this.writer.write(this.makeTagStart("maxPool") + ruleServiceApp.getMaxPool() + this.makeTagEnd("maxPool") + LINE_BREAK);
		this.writeTabs();
		this.writer.write(this.makeTagStart("minPool") + ruleServiceApp.getMinPool() + this.makeTagEnd("minPool") + LINE_BREAK);
		this.writeTabs();
		this.writer.write(this.makeTagStart("initialPool") + ruleServiceApp.getInitialPool() + this.makeTagEnd("initialPool") + LINE_BREAK);
		this.writeTabs();
		this.writer.write(this.makeTagStart("currentPoolCount") + ruleServiceApp.getCurrentPoolCount() + this.makeTagEnd("currentPoolCount") + LINE_BREAK);
		
		this.popTab();
		
		this.writeTabs();
		this.writer.write(this.makeTagEnd(ruleServiceApp.getClass().getName()) + LINE_BREAK);
	}
	
	protected void writeModels(Object model) throws IOException {
		if (model instanceof ObjectModel) {
			ObjectModel objModel = (ObjectModel)model;
			this.writeTabs();
			this.writer.write(this.makeTagStart(objModel.getClass().getName()) + LINE_BREAK);
			this.pushTab();
			
			this.writeTabs();
			if (objModel.getURL() != null) {
				this.writer.write(this.makeTagStart("URL") + objModel.getURL() + this.makeTagEnd("URL") + LINE_BREAK);
			} else {
				this.writer.write(this.makeEmptyTag("URL") + LINE_BREAK);
			}
			
			this.writeTabs();
			this.writer.write(this.makeTagStart("classList") + LINE_BREAK);
			this.pushTab();
			for (int i=0; i < objModel.getClassList().size(); i++) {
				String clss = (String)objModel.getClassList().get(i);
				this.writeTabs();
				this.writer.write(this.makeTagStart("string") + clss + this.makeTagEnd("string") + LINE_BREAK);
			}
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd("classList") + LINE_BREAK);
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd(objModel.getClass().getName()) + LINE_BREAK);
		}
	}
	
	/**
	 * method writes out the InitialData. Currently it only handles clips initial data.
	 * It will eventually need to support object data.
	 * @param data
	 * @throws IOException
	 */
	protected void writeInitialData(Object data) throws IOException {
		if (data instanceof ClipsInitialData) {
			ClipsInitialData clipsData = (ClipsInitialData)data;
			this.writeTabs();
			this.writer.write(this.makeTagStart(clipsData.getClass().getName()) + LINE_BREAK);
			this.pushTab();
			
			this.writeTabs();
			if (clipsData.getURL() != null) {
				this.writer.write(this.makeTagStart("URL") + clipsData.getURL() + this.makeTagEnd("URL") + LINE_BREAK);
			} else {
				this.writer.write(this.makeEmptyTag("URL") + LINE_BREAK);
			}
			this.writeTabs();
			if (clipsData.getName() != null) {
				this.writer.write(this.makeTagStart("name") + clipsData.getName() + this.makeTagEnd("name") + LINE_BREAK);
			} else {
				this.writer.write(this.makeEmptyTag("name") + LINE_BREAK);
			}
			
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd(clipsData.getClass().getName()) + LINE_BREAK);
		}
	}
	
	/**
	 * 
	 * @param data
	 * @throws IOException
	 */
	protected void writeFunctionGroup(Object data) throws IOException {
		if (data instanceof FunctionPackage) {
			FunctionPackage fncPkg = (FunctionPackage)data;
			this.writeTabs();
			this.writer.write(this.makeTagStart(fncPkg.getClass().getName()) + LINE_BREAK);
			this.pushTab();
			
			this.writeTabs();
			if (fncPkg.getURL() != null) {
				this.writer.write(this.makeTagStart("URL") + fncPkg.getURL() + this.makeTagEnd("URL") + LINE_BREAK);
			} else {
				this.writer.write(this.makeEmptyTag("URL") + LINE_BREAK);
			}
			this.writeTabs();
			this.writer.write(this.makeTagStart("classNames") + LINE_BREAK);
			this.pushTab();
			for (int i=0; i < fncPkg.getClassNames().length; i++) {
				String clss = fncPkg.getClassNames()[i];
				this.writeTabs();
				this.writer.write(this.makeTagStart("string") + clss + this.makeTagEnd("string") + LINE_BREAK);
			}
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd("classNames") + LINE_BREAK);
			
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd(fncPkg.getClass().getName()) + LINE_BREAK);
		}
	}
	
	protected void writeRuleset(Object data) throws IOException {
		if (data instanceof ClipsRuleset) {
			ClipsRuleset clipsRuleset = (ClipsRuleset)data;
			this.writeTabs();
			this.writer.write(this.makeTagStart(clipsRuleset.getClass().getName()) + LINE_BREAK);
			this.pushTab();
			
			this.writeTabs();
			if (clipsRuleset.getURL() != null) {
				this.writer.write(this.makeTagStart("URL") + clipsRuleset.getURL() + this.makeTagEnd("URL") + LINE_BREAK);
			} else {
				this.writer.write(this.makeEmptyTag("URL") + LINE_BREAK);
			}
			this.writeTabs();
			this.writer.write(this.makeTagStart("contents") + clipsRuleset.getContents() + this.makeTagEnd("contents") + LINE_BREAK);
			
			this.popTab();
			this.writeTabs();
			this.writer.write(this.makeTagEnd(clipsRuleset.getClass().getName()) + LINE_BREAK);
		}
	}
	
	protected String makeTagStart(String name) {
		return TAGSTART + name + TAGEND;
	}
	
	protected String makeTagEnd(String name) {
		return TAGSTART + "/" + name + TAGEND;
	}
	
	protected String makeEmptyTag(String name) {
		return TAGSTART + name + "/" + TAGEND;
	}
	
	protected void pushTab() {
		this.tabCount++;
	}
	
	protected void popTab() {
		this.tabCount--;
	}
	
	protected void writeTabs() {
		try {
			for (int i=0; i < this.tabCount; i++) {
				this.writer.write(this.TAB);
			}
		} catch (IOException e) {
		}
	}
}
