/*
 * Copyright 2002-2008 Peter Lin
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://jamocha.sourceforge.net/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 */
package org.jamocha.rete;

import java.io.Serializable;

/**
 * @author Peter Lin
 *
 * A binding can be an object or a field of a given object. When binding
 * is an Object binding, the column id will not be set. The binding
 * would be the row for the left memory.</br>
 * It is up to classes using the binding to check if it is an object
 * binding and get the appropriate fact using getLeftRow().
 * One thing about the current design is the binding is position based.
 * The benefit is it avoids having to set the binding and reset it
 * multiple times. BetaNodes use the binding to get the correct slot
 * value and use it to evaluate an atomic condition. A significant
 * downside of this approach is when deftemplates are re-declared at
 * runtime. It means that we might need to recompute the bindings, which
 * could be a very costly process. More thought and research is needed
 * to figure out the best way to handle re-declaring deftemplates.
 */
public class Binding implements Serializable, Cloneable {

    /**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
     * This is the name of the variable. Every binding must
     * have a variable name. It can be user defined or auto-
     * generated by the rule compiler.
     */
    protected String varName = null;

    /**
     * if the binding is to an object, the field should
     * be true. by default it is false.
     */
    protected boolean isObjVar = false;
    
    /**
     * if the join is a predicate join with a function, the
     * binding should be set to isPredicate
     */
    protected boolean isPredJoin = false;
    
    /**
     * by default the row index is -1, which means
     * it's not set. Any index that is negative indicates
     * it's not set.
     */
    protected int leftrow = -1;
    
    /**
     * The indexes of the left deftemplate
     */
    protected int leftIndex;

    /**
     * the indexes for the right deftemplate
     */
    protected int rightIndex;
    
    /**
     * by default bindings test for equality. in some cases
     * they test for inequality.
     */
    protected boolean negated = false;
    
    /**
     * We need this to keep track of which CE is the first to declare
     * a binding. This is important to rule compilation.
     */
    protected int rowDeclared = -1;
    
    public Binding(){
        super();
    }
    
    /**
     * Return the name of the variable
     * @return
     */
    public String getVarName(){
        return this.varName;
    }
    
    /**
     * Set the variable name. This is important, since the join
     * nodes will use it at runtime.
     * @param name
     */
    public void setVarName(String name){
        this.varName = name;
    }

    /**
     * If the binding is for an object, the method returns true.
     * @return
     */
    public boolean getIsObjectVar(){
        return this.isObjVar;
    }

    /**
     * Set whether the binding is an object binding.
     * @param obj
     */
    public void setIsObjectVar(boolean obj){
        this.isObjVar = obj;
    }
    
    /**
     * The row that declares the binding the first time. The
     * row corresponds directly to the Conditional Element in
     * the rule. If the second CE declares the binding for the
     * first time, the row would be 1. 
     * @return
     */
    public int rowDeclared() {
    	return this.rowDeclared;
    }
    
    /**
     * Set the row that declares the binding
     * @param row
     */
    public void setRowDeclared(int row) {
    	this.rowDeclared = row;
    }
    
    /**
     * Return the left Deftemplate 
     * @return
     */
    public int getLeftRow(){
        return this.leftrow;
    }

    /**
     * Set the left deftemplate
     * @param temp
     */
    public void setLeftRow(int left){
        this.leftrow = left;
    }
    
    /**
     * get the left index
     * @return
     */
    public int getLeftIndex(){
        return this.leftIndex;
    }

    /**
     * set the left index
     * @param indx
     */
    public void setLeftIndex(int indx){
        this.leftIndex = indx;
    }
    
    /**
     * get the right index
     * @return
     */
    public int getRightIndex(){
        return this.rightIndex;
    }
    
    /**
     * set the right index
     * @param indx
     */
    public void setRightIndex(int indx){
        this.rightIndex = indx;
    }
    
    /**
     * by default bindings are not negated. if it is,
     * method return true.
     * @return
     */
    public boolean negated() {
    	return this.negated;
    }
    
    /**
     * if a binding is negated, call the method with true
     * @param neg
     */
    public void setNegated(boolean neg) {
    	this.negated = neg;
    }
    
    /**
     * Since the binding refers to the row and column, the binding
     * doesn't know the deftemplate.
     * @return
     */
    public String toBindString(){
        StringBuffer buf = new StringBuffer();
        buf.append("(" + this.leftrow + ")(");
        buf.append(this.leftIndex);
        if (this.negated) {
            buf.append(") " + Constants.NOTEQUAL_STRING + " (0)(");
        } else {
            buf.append(") " + Constants.EQUAL_STRING + " (0)(");
        }
        buf.append(this.rightIndex);
        buf.append(")");
        return buf.toString();
    }
    
    public String toPPString() {
        StringBuffer buf = new StringBuffer();
        buf.append("?" + varName + " (" + this.leftrow + ")(");
        buf.append(this.leftIndex);
        if (this.negated) {
            buf.append(") " + Constants.NOTEQUAL_STRING + " (0)(");
        } else {
            buf.append(") " + Constants.EQUAL_STRING + " (0)(");
        }
        buf.append(this.rightIndex);
        buf.append(")");
        return buf.toString();
    }
    
    /**
     * convienance method for clonging a binding at rule compilation
     * time.
     */
    public Object clone(){
        Binding bind = new Binding();
        bind.setVarName(this.getVarName());
        bind.setIsObjectVar(this.getIsObjectVar());
        bind.setLeftRow(this.getLeftRow());
        bind.setLeftIndex(this.getLeftIndex());
        bind.setRightIndex(this.getRightIndex());
        return bind;
    }
    
    /**
     * evaluate will extra the values from each side and evaluate it
     * @param left
     * @param right
     * @return
     */
    public boolean evaluate(Fact[] left, Fact right) {
        if (left[leftrow] == right) {
            return false;
        } 
        if (negated) {
            return Evaluate.evaluateNotEqual(left[leftrow].getSlotValue(leftIndex),right.getSlotValue(rightIndex));
        } else {
            return Evaluate.evaluateEqual(left[leftrow].getSlotValue(leftIndex),right.getSlotValue(rightIndex));
        }
    }

	public void setPredJoin(boolean isPredJoin) {
		this.isPredJoin = isPredJoin;
	}
}
