package jadex.tools.testcenter;

import jadex.config.Configuration;
import jadex.model.*;
import jadex.tools.starter.*;
import jadex.util.SGUI;

import javax.swing.*;
import javax.swing.tree.TreeNode;
import java.io.File;
import java.io.IOException;

/**
 *  A node representing a file (overrides valid functionality and icons).
 */
public class TestFileNode extends FileNode
{
	//-------- constants --------

	/**
	 * The image  for (m/r) elements.
	 */
	static UIDefaults icons = new UIDefaults(new Object[]
	{
		"agent", SGUI.makeIcon(TestFileNode.class, "/jadex/tools/common/images/new_agent.png"),
		"agent_testable", SGUI.makeIcon(TestFileNode.class, "/jadex/tools/common/images/new_agent_testable.png"),
	});


	//-------- constructors --------

	/**
	 *  Create a new test file node.
	 *  @param parent
	 *  @param file
	 */
	public TestFileNode(TreeNode parent, File file)
	{
		super(parent, file);
		setValid(false);
	}
	
	/**
	 *  Create a new test file node.
	 *  Bean constructor.
	 */
	public TestFileNode()
	{
	}	
	
	//-------- methods --------

	/**
	 *  Check if this object is equal to another object.
	 */
	public boolean equals(Object obj)
	{
		return (obj instanceof TestFileNode) && this.file.getAbsolutePath().equals(((TestFileNode)obj).file.getAbsolutePath());
	}

	/**
	 *  Actualy perform the check.
	 *  Can be overridden by subclasses.
	 *  @return True, if the node has changed since last check (if any).
	 */
	protected boolean doCheck()
	{
		boolean	newvalid	= false;
		String	file	= getFile().getAbsolutePath();
		if(SXML.isJadexFilename(file))
		{
			try
			{
				// todo: remove Hack! Let tree always load fresh models when autorefresh is off
				if(Configuration.getConfiguration().isModelCaching() && !Configuration.getConfiguration().isModelCacheAutoRefresh())
					SXML.clearModelCache(file);
				
				IMBDIAgent	model = null;
				if(SXML.isAgentFilename(file))
					model	= SXML.loadAgentModel(file, null);
				
				if(model!=null)
				{
					boolean ok	= !jadex.config.Configuration.getConfiguration().isModelChecking()
						|| model.getReport().isEmpty();

					if(ok)
					{
						IMCapabilityReference[] caprefs = model.getCapabilityReferences();
						for(int i=0; !newvalid && i<caprefs.length; i++)
						{
							String name = caprefs[i].getCapability().getFullName();
							newvalid = name.equals("jadex.planlib.Test");
						}
					}
				}
				// else unknown jadex file type -> ignore.
			}
			catch(IOException e)
			{
			}
		}

		// Update valid state.
		boolean	changed	= isValid()!=newvalid;
		setValid(newvalid);
		return changed;
	}
	
	/**
	 *  Return the valid state of the node.
	 */
	public boolean getState()
	{
		return isValid();
	}

	/**
	 * @return the icon for this node
	 * @see jadex.tools.starter.IExplorerTreeNode#getIcon()
	 */
	public Icon getIcon()
	{
		Icon	icon	= null;
		if(SXML.isAgentFilename(file.getName()))
		{
			icon	= icons.getIcon(getState() ? "agent_testable": "agent");
		}
		return icon;
	}

	/**
	 *  Return the tooltip text for the node (if any).
	 */
	// todo: remove
	public String getToolTipText()
	{
		return getFile().getAbsolutePath();
	}
}
