package jadex.tools.starter;

import java.io.File;
import java.util.*;

import javax.swing.Icon;
import javax.swing.tree.TreeNode;

/**
 *  A directory node.
 */
public class DirNode	extends FileNode
{
	//-------- attributes --------

	/** The children of the node (i.e. contained files and subdirectories). */
	protected List	children;

	//-------- constructors --------

	/**
	 *  Create a directory node for a given directory.
	 */
	public DirNode(TreeNode parent, File dir)
	{
		super(parent, dir);
	}

	/**
	 *  Create a new DirNode.
	 *  Bean constructor.
	 */
	public DirNode()
	{
	}

	//-------- TreeNode interface --------

	/**
	 *  Returns the child TreeNode at index childIndex.
	 *  @param childIndex the index of the child to return
	 *  @return a TreeNode instance
	 */
	public TreeNode getChildAt(int childIndex)
	{
		if(children==null)
			throw new ArrayIndexOutOfBoundsException(childIndex);
		return (TreeNode)children.get(childIndex);
	}

	/**
	 *  Returns the number of children TreeNodes the receiver contains.
	 *  @return the number of children TreeNodes the receiver contains
	 */
	public int getChildCount()
	{
		return children!=null ? children.size() : 0;
	}

	/**
	 *  Returns the index of node in the receivers children. If the receiver
	 *  does not contain node, -1 will be returned.
	 *  @param node
	 *  @return an int.
	 */
	public int getIndex(TreeNode node)
	{
		return children!=null ? children.indexOf(node) : -1;
	}

	/**
	 *  Returns true if the receiver allows children.
	 *  @return a boolean.
	 */
	public boolean getAllowsChildren()
	{
		return true;
	}

	/**
	 *  Returns true if the receiver is a leaf.
	 *  @return a boolean
	 */
	public boolean isLeaf()
	{
		return false;
	}

	/**
	 *  Returns the children of the reciever as an Enumeration.
	 *  @return an Enumeration
	 */
	public Enumeration children()
	{
		return children!=null?Collections.enumeration(children):EMPTY_ENUMERATION;
	}
	
	//-------- methods --------

	/**
	 *  Perform the actual refresh (if necessary).
	 *  Updates the children of this node
	 *  and sorts them, first directory
	 *  then files in alphabetical order.
	 */
	protected boolean	doRefresh()
	{
		boolean	changed	= super.doRefresh();

		if(changed)
		{
//			System.out.println("refreshing: "+this);
			File files[] = getFile().listFiles(getRootNode().getFilter());
			if(files!=null)
			{
				Set	old	= null;
				if(children!=null)
				{
					old	= new HashSet(children);
				}
				else if(files.length>0)
				{
					children	= new ArrayList();
				}
				
				for(int i = 0; i<files.length; i++)
				{
					TreeNode child = createNode(files[i]);
	
					// Check if child is new
					if(old==null || !old.remove(child))
					{
						int	index;
						for(index=0; index<children.size() && FILENODE_COMPARATOR.compare(children.get(index), child)<=0; index++);
						children.add(index, child);
					}
				}
				
				// Remove old entries.
				if(old!=null)
				{
					for(Iterator it=old.iterator(); it.hasNext(); )
					{
						children.remove(it.next());
					}
				}
			}
			
			// Cannot access directory.
			else if(children!=null)
			{
				children	= null;
			}
		}

		return changed;
	}

	/**
	 *  Check the state of the node.
	 *  @return true, when a change has been detected.
	 */
	protected boolean	doCheck()
	{
		boolean	newvalid	= true;
		for(int i=0; newvalid	&& i<getChildCount(); i++)
			newvalid	= newvalid && ((FileNode)getChildAt(i)).getState();
		
//		System.out.println("doCheck "+this+": "+isValid()+"->"+newvalid);
		
		// Update valid state.
		boolean	changed	= isValid()!=newvalid;
		if(changed)
		{
			setValid(newvalid);			
		}
		return changed;
	}

	/**
	 *  Create a node for a given file.
	 */
	protected TreeNode createNode(File file)
	{
		return file.isDirectory()
			? (TreeNode)new DirNode(this, file)
			: (TreeNode)new FileNode(this, file);
	}	

	/** 
	 * @return the icon for this node
	 * @see jadex.tools.starter.FileNode#getIcon()
	 */
	public Icon getIcon()
	{
		Icon	icon	= null;
		if(getParent() instanceof RootNode)
		{
			icon	= ModelTreeCellRenderer.icons.getIcon(getState() ? "src_folder" : "src_folder_broken");
		}
		else
		{
			icon	= ModelTreeCellRenderer.icons.getIcon(getState() ? "package" : "package_broken");
		}
		return icon;
	}

	/**
	 * Return the string reoresentation of this node.
	 * @return its name
	 */
	public String toString()
	{
		return (getParent() instanceof RootNode)? file.getName()+" ("+file.getAbsolutePath()+")": file.getName();
	}

	public static final Comparator FILENODE_COMPARATOR = new Comparator()
	{
		public int compare(Object o1, Object o2)
		{
			File f1 = ((FileNode)o1).getFile();
			File f2 = ((FileNode)o2).getFile();
			if(f1.isDirectory() && !f2.isDirectory()) return -1;
			if(!f1.isDirectory() && f2.isDirectory()) return 1;

			return f1.getName().compareTo(f2.getName());
		}
	};


	//-------- bean property accessors --------
	
	/**
	 *  Get the children of this DirNode.
	 *  @return Returns the children.
	 */
	public List getChildren()
	{
		return children;
	}

	/**
	 *  Set the children of this DirNode.
	 *  @param children The children to set.
	 */
	public void setChildren(List children)
	{
		this.children = children;
	}
}
