package jadex.tools.dfbrowser;

import jadex.adapter.fipa.*;
import jadex.runtime.IGoal;
import jadex.tools.common.*;
import jadex.tools.common.jtreetable.AbstractTreeTableAction;
import jadex.tools.common.jtreetable.DefaultTreeTableNode;
import jadex.tools.common.plugin.IAgentListListener;
import jadex.tools.jcc.AbstractJCCPlugin;
import jadex.tools.starter.StarterPlugin;
import jadex.util.SGUI;

import javax.swing.*;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import java.awt.*;
import java.awt.event.*;
import java.util.Properties;


/**
 * DFBrowserPlugin
 */
public class DFBrowserPlugin extends AbstractJCCPlugin implements IAgentListListener
{
	//-------- constants --------
	
	/** The image icons. */
	protected static final UIDefaults icons = new UIDefaults(new Object[]
	{
		"dfbrowser", SGUI.makeIcon(DFBrowserPlugin.class, "/jadex/tools/common/images/new_dfbrowser.png"), 
		"dfbrowser_sel", SGUI.makeIcon(DFBrowserPlugin.class, "/jadex/tools/common/images/new_dfbrowser_sel.png"), 
		"remove_agent", SGUI.makeIcon(DFBrowserPlugin.class, "/jadex/tools/common/images/new_remove_service.png"),
		"starter", SGUI.makeIcon(StarterPlugin.class, "/jadex/tools/common/images/new_starter.png"),
	});

	/** refresh every second */
	protected static final long REFRESH1 = 1000;

	/** refresh every 5 seconds */
	protected static final long REFRESH5 = 5000;

	/** refresh every 30 seconds */
	protected static final long REFRESH30 = 30000;
	
	//-------- attributes --------
	
	/** The agent table (showing platform DFs). */
	protected AgentTreeTable df_agents;

	/** The agent table. */
	protected DFAgentTable agent_table;

	/** The service table. */
	protected DFServiceTable service_table;

	/** The service panel. */
	protected ServiceDescriptionPanel service_panel;

	/** How long should the refresh process wait */
	protected long sleep = REFRESH5;

	/** Current df */
	protected AMSAgentDescription df_des;

	/** The thread that refreshes the plugin views */
	protected volatile Thread refresh_thread;

	/** The main panel. */
	protected JPanel main_panel;

	/** The first split pane. */
	protected JSplitPane split1;

	/** The second split pane. */
	protected JSplitPane split2;

	/** The third split pane. */
	protected JSplitPane split3;

	/** Refresh setting . */
	protected JRadioButtonMenuItem refresh1;

	/** Refresh setting . */
	protected JRadioButtonMenuItem refresh5;

	/** Refresh setting . */
	protected JRadioButtonMenuItem refresh30;

	/** selected_agent */
	protected AgentDescription selected_agent;

	//-------- methods --------
	
	/**
	 * @return "DF Browser"
	 * @see jadex.tools.common.plugin.IControlCenterPlugin#getName()
	 */
	public String getName()
	{
		return "DF Browser";
	}

	/**
	 * @return the icon of DFBrowser
	 * @see jadex.tools.common.plugin.IControlCenterPlugin#getToolIcon()
	 */
	public Icon getToolIcon(boolean selected)
	{
		return selected? icons.getIcon("dfbrowser_sel"): icons.getIcon("dfbrowser");
	}

	/**
	 * @return the tool bar
	 */
	public JToolBar getToolBar()
	{
		JToolBar bar = new JToolBar();
		JButton b;

		b = new JButton(REMOVE_AGENT);
		b.setBorder(null);
		b.setToolTipText(b.getText());
		b.setText(null);
		b.setEnabled(true);
		bar.add(b);

		return bar;
	}

	/**
	 * @return the central panel of DFBrowser
	 */
	public JComponent getPanel()
	{
		df_agents = new AgentTreeTable();
		df_agents.setMinimumSize(new Dimension(0, 0));
		df_agents.getTreetable().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		df_agents.getTreetable().getSelectionModel().addListSelectionListener(
		new ListSelectionListener()
		{
			public void valueChanged(ListSelectionEvent e)
			{
				JTree tree = df_agents.getTreetable().getTree();
				if(!e.getValueIsAdjusting() && !tree.isSelectionEmpty())
				{
					SHOW_DF.actionPerformed(null);
				}
			}
		});
		df_agents.getNodeType(AgentTreeTable.NODE_AGENT).addPopupAction(SHOW_DF);

		df_agents.getTreetable().getSelectionModel().setSelectionInterval(0, 0);

		df_agents.getTreetable().addMouseListener(new MouseAdapter()
		{
			/**
			 * will start DFBrowser
			 *
			 * @param e
			 */
			public void mouseClicked(MouseEvent e)
			{
				if(e.getClickCount() == 2)
				{
					SHOW_DF.actionPerformed(null);
				}

			}
		});

		jcc.addAgentListListener(this);
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run()
			{
				df_agents.adjustColumnWidths();
			}
		});

		service_panel = new ServiceDescriptionPanel();
		service_table = new DFServiceTable()
		{
			/**
			 * @param service_description
			 * @param agent_description
			 */
			protected void serviceSelected(ServiceDescription service_description,
					AgentDescription agent_description)
			{
				service_panel.setService(agent_description, service_description);
			}

		};
		agent_table = new DFAgentTable(this)
		{

			/**
			 * @param agent_description
			 * @see jadex.tools.dfbrowser.DFAgentTable#agentSelected(jadex.adapter.fipa.AgentDescription)
			 */
			protected void agentSelected(AgentDescription agent_description)
			{
				selected_agent = agent_description;
				service_table.setAgentDescription(agent_description);
			}

			/**
			 * @param agent_description
			 */
			protected void removeSelected(AgentDescription agent_description)
			{
				removeAgentRegistration(agent_description);
			}
		};

		main_panel = new JPanel(new BorderLayout());
		
		split3 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		split3.setDividerLocation(250);
		split3.add(service_table);
		split3.add(service_panel);
		split2 = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		split2.setDividerLocation(250);
		split2.add(agent_table);
		split2.add(split3);
		split1 = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
		split1.setDividerLocation(200);
		split1.setOneTouchExpandable(true);
		split1.add(df_agents);
		split1.add(split2);
	
		main_panel.removeAll();
		main_panel.add(split1, BorderLayout.CENTER);
		main_panel.validate();

		GuiProperties.setupHelp(split1, "tools.DFBrowser");

		return main_panel;
	}

	/**
	 * Load the properties.
	 *
	 * @param props
	 */
	public void setProperties(Properties props)
	{
		// try to load arguments from properties.
		try
		{
			split3.setDividerLocation(Integer.parseInt(props.getProperty("split3.location")));
		}
		catch(Exception e)
		{ /* NOP */
		}
		try
		{
			split2.setDividerLocation(Integer.parseInt(props.getProperty("split2.location")));
		}
		catch(Exception e)
		{ /* NOP */
		}
		try
		{
			split1.setDividerLocation(Integer.parseInt(props.getProperty("split1.location")));
		}
		catch(Exception e)
		{ /* NOP */
		}
		try
		{
			REFRESH.setSelected("true".equals(props.getProperty("refresh")));
		}
		catch(Exception e)
		{ /* NOP */
		}
		
		agent_table.setProperties(props);
		
		service_table.setProperties(props);
		
		if(refresh1 != null)
		{
			refresh1.setSelected(sleep == REFRESH1);
		}
		if(refresh5 != null)
		{
			refresh5.setSelected(sleep == REFRESH5);
		}
		if(refresh30 != null)
		{
			refresh30.setSelected(sleep == REFRESH30);
		}
	}

	/**
	 * Save the properties.
	 *
	 * @param props
	 */
	public void getProperties(Properties props)
	{
		props.put("split1.location", Integer.toString(split1.getDividerLocation()));
		props.put("split2.location", Integer.toString(split2.getDividerLocation()));
		props.put("split3.location", Integer.toString(split3.getDividerLocation()));
		//props.put("layout", Integer.toString(layout));
		props.put("refresh", Boolean.toString(REFRESH.isSelected()));
		agent_table.getProperties(props);
		service_table.getProperties(props);
	}

	/**
	 * @return the modified menubar
	 */
	public JMenuBar createMenuBar()
	{
		JMenuBar menubar = new JMenuBar();
		JMenu menu;

		menu = new JMenu("View");
		GuiProperties.setupHelp(menu, "df_browser.menu_graph");

		menubar.add(menu);
		menu.add(REFRESH.getCBItem());
		ButtonGroup group = new ButtonGroup();
		refresh1 = new JRadioButtonMenuItem(new AbstractAction("1 s")
		{
			public void actionPerformed(ActionEvent e)
			{
				sleep = REFRESH1;
			}
		});
		refresh1.setSelected(sleep == REFRESH1);
		group.add(refresh1);
		menu.add(refresh1);

		refresh5 = new JRadioButtonMenuItem(new AbstractAction("5 s")
		{
			public void actionPerformed(ActionEvent e)
			{
				sleep = REFRESH5;
			}
		});
		refresh5.setSelected(sleep == REFRESH5);
		group.add(refresh5);
		menu.add(refresh5);

		refresh30 = new JRadioButtonMenuItem(new AbstractAction("30 s")
		{
			public void actionPerformed(ActionEvent e)
			{
				sleep = REFRESH30;
			}
		});
		refresh30.setSelected(sleep == REFRESH30);
		group.add(refresh30);
		menu.add(refresh30);

		menu.addSeparator();

		return menubar;
	}

	AgentDescription[] old_ads;

	/**
	 * 
	 */
	protected void refresh()
	{
		if(df_des != null)
		{
			SearchConstraints constraints = new SearchConstraints();
			constraints.setMaxResults(-1);

			// Use a subgoal to search
			IGoal ft = getJCC().getAgent().createGoal("df_search");
			ft.getParameter("description").setValue(new AgentDescription());
			ft.getParameter("constraints").setValue(constraints);
			ft.getParameter("df").setValue(df_des.getName());

			getJCC().getAgent().dispatchTopLevelGoalAndWait(ft);
			AgentDescription[] ads = (AgentDescription[])ft.getParameterSet("result").getValues();

			if(old_ads == null || !equal(old_ads, ads))
			{
				old_ads = ads;
				agent_table.setAgentDescriptions(ads);
				if(selected_agent == null)
				{ // show all services
					service_table.setAgentDescriptions(ads);
				}
				else
				{
					showOnlyServicesFromSelectedAgent(ads);
				}
			}
		}
	}

	/**
	 * @param ads
	 */
	protected void showOnlyServicesFromSelectedAgent(AgentDescription[] ads)
	{
		AgentIdentifier aid = selected_agent.getName();
		for(int i = 0; i < ads.length; i++)
		{
			if(aid.equals(ads[i].getName()))
			{
				service_table.setAgentDescription(ads[i]);
				return;
			}
		}
		// agent disappeared, show all services anyway
		service_table.setAgentDescriptions(ads);
	}

	/**
	 * @param ad1
	 * @param ad2
	 * @return true if both tables contain same descriptions
	 */
	protected static boolean equal(AgentDescription[] ad1, AgentDescription[] ad2)
	{
		if(ad1.length != ad2.length)
		{
			return false;
		}
		for(int i = 0; i < ad1.length; i++)
		{
			if(!ad1[i].equals(ad2[i]))
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * @param ad
	 */
	public void agentDied(final AMSAgentDescription ad)
	{
		if(df_des != null && df_des.getName().equals(ad.getName()))
		{
			df_des = null;
		}
		// Update components on awt thread.
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run()
			{
				df_agents.removeAgent(ad);
			}
		});
	}

	/**
	 * @param ad
	 */
	public void agentBorn(final AMSAgentDescription ad)
	{
		// Update components on awt thread.
		SwingUtilities.invokeLater(new Runnable()
		{
			public void run()
			{
				// todo: Megahack replace with sth. useful (agent need not be there,
				// todo: other agents could be the df and not be named df?)
				if(ad.getName().getName().startsWith("df@"))
				{
					df_agents.addAgent(ad);
					if(df_des == null)
					{
						df_des = ad;
						REFRESH.flagChanged(REFRESH.isSelected());
					}
				}
			}
		});
	}
	
	/**
	 * @param ad
	 */
	public void agentChanged(final AMSAgentDescription ad)
	{
		// nop?
		// Update components on awt thread.
		/*SwingUtilities.invokeLater(new Runnable()
		{
			public void run()
			{
				agents.addAgent(ad);
			}
		});*/
	}

	final AbstractTreeTableAction SHOW_DF = new ShowDFAction();

	protected final class ShowDFAction extends AbstractTreeTableAction
	{
		protected ShowDFAction()
		{
			super("Show DF", icons.getIcon("DFBrowser"));
		}

		/**
		 * @param e
		 * @see jadex.tools.common.jtreetable.AbstractTreeTableAction#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			DefaultTreeTableNode node = (DefaultTreeTableNode)df_agents.getTreetable().getTree()
					.getSelectionPath().getLastPathComponent();
			if(node != null && node.getUserObject() instanceof AMSAgentDescription)
			{
				main_panel.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				if(selected_agent != null)
				{
					selected_agent = null;
					old_ads = null;
				}
				DFBrowserPlugin.this.df_des = (AMSAgentDescription)node.getUserObject();
				DFBrowserPlugin.this.refresh();
				main_panel.setCursor(Cursor.getDefaultCursor());
			}
		}

		/**
		 * @return false
		 * @see jadex.tools.common.jtreetable.AbstractTreeTableAction#isSelected()
		 */
		public boolean isSelected()
		{
			DefaultTreeTableNode node = (DefaultTreeTableNode)getValue("node");
			return node.getUserObject() == df_des;
		}
	}


	/**
	 * @return the help id of the perspective
	 * @see jadex.tools.jcc.AbstractJCCPlugin#getHelpID()
	 */
	public String getHelpID()
	{
		return "tools.df_browser";
	}

	/**
	 * 
	 */
	protected void startRefreshThread()
	{
		refresh_thread = new Thread("DFBrowser refresh thread")
		{
			public void run()
			{
				try
				{
					while(REFRESH.isSelected() && refresh_thread == this)
					{
						refresh();
						sleep(DFBrowserPlugin.this.sleep);
					}
				}
				catch(Exception e)
				{//
				}
				if(refresh_thread == this)
				{
					refresh_thread = null;
				}
			}
		};
		refresh_thread.start();
	}

	/**
	 * @see jadex.tools.jcc.AbstractJCCPlugin#shutdown()
	 */
	public void shutdown()
	{
		refresh_thread = null;
	}

	/**
	 * @param description
	 */
	protected void removeAgentRegistration(AgentDescription description)
	{
		try
		{
			IGoal deregister = getJCC().getAgent().createGoal("df_deregister");
			deregister.getParameter("description").setValue(description);
			deregister.getParameter("df").setValue(df_des.getName());
			getJCC().getAgent().dispatchTopLevelGoalAndWait(deregister, 100);
			refresh();
		}
		catch(Exception e)
		{
			// NOP
		}
	}
	
	/**
	 * should df descriptions be refreshed
	 */
	protected SelectAction REFRESH = new SelectAction("Refresh")
	{
		public void flagChanged(boolean flag)
		{
			if(refresh_thread == null && flag)
			{
				// todo: remove extra thread!
				startRefreshThread();
			}
		}
	};
	
	/**
	 *  Removes agent from the df
	 */
	final AbstractAction REMOVE_AGENT = new RemoveAgentAction();

	/**
	 *  The remove agent action. 
	 */
	protected final class RemoveAgentAction extends AbstractAction
	{
		protected RemoveAgentAction()
		{
			super("Remove agent registration", icons.getIcon("remove_agent"));
		}

		/**
		 * @param e
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if(df_des != null)
			{
				AgentDescription[] as = agent_table.getSelectedAgents();
				for(int i = 0; i < as.length; i++)
				{
					removeAgentRegistration(as[i]);
				}
				refresh();
			}
		}
	}
}
