package jadex.planlib;

import jadex.adapter.fipa.AgentIdentifier;
import jadex.adapter.fipa.SFipa;
import jadex.runtime.IMessageEvent;

import java.text.SimpleDateFormat;
import java.util.*;

/**
 *  The negotiation record has the purpose to store all information
 *  about one negotiation round in an iterated contract net protocol.
 */
public class NegotiationRecord
{
	//-------- constants --------
	
	/** The negotiation is not yet finished. */
	public static final String OPEN = "Open";
	
	/** The negotiation has finished successfully. */
	public static final String SUCCEEDED = "succeeded";
	
	/** The negotiation is finished with a failure. */
	public static final String FAILED = "failed";
	
	//-------- variables --------

	/** The cfp sent to the participants. */
	protected Object cfp;

	/** All proposals sent from the participants. */
	protected Map proposals;

	/** Acceptable proposals, form a subset of received proposals. */
	protected List acceptables;

	/** Winner proposals, form a subset of acceptable proposals. */
	protected List winners;

	/** The executed tasks. */
	protected List tasks;
	
	/** The negotiation status (open, when last round succeeded or failed). */
	protected String state;
	
	/** The start time. */
	protected long starttime;
	
	/** The end time. */
	protected long endtime;


	//-------- constructors --------

	/**
	 *  Create a new negotiation record.
	 *  @param cfp The call for proposal.
	 */
	public NegotiationRecord(Object cfp)
	{
		this.cfp = cfp;
		this.proposals = new HashMap();
		this.acceptables = new ArrayList();
		this.winners = new ArrayList();
		this.tasks = new ArrayList();
		this.state = OPEN;
	}

	//-------- methods --------

	/**
	 *  Add a proposal (and the message).
	 *  @param msg The proposal message.
	 */
	public void addProposal(IMessageEvent msg)
	{
		proposals.put(msg.getContent(),  new ProposalInfo(msg));
	}

	/**
	 *  Set the acceptable proposals.
	 *  @param proposals The acceptable proposals.
	 */
	public void setAcceptableProposals(Object[] proposals)
	{
		for(int i=0; i<proposals.length; i++)
			acceptables.add(proposals[i]);
	}

	/**
	 *  Set the winner proposals.
	 *  @param proposals The winner proposals.
	 */
	public void setWinnerProposals(Object[] proposals)
	{
		for(int i=0; i<proposals.length; i++)
			winners.add(proposals[i]);
	}

	/**
	 *  Add an executed task.
	 *  @param task The executed task.
	 */
	public void addExecutedTask(Object task)
	{
		tasks.add(task);
	}

	/**
	 *  Get all proposals.
	 *  @return All proposals.
	 */
	public Object[] getProposals()
	{
		return proposals.keySet().toArray();
	}

	/**
	 *  Get the acceptable proposals.
	 *  @return The acceptable proposals.
	 */
	public Object[] getAcceptableProposals()
	{
		return acceptables.toArray();
	}

	/**
	 *  Get the winner proposals.
	 *  @return The winner proposals.
	 */
	public Object[] getWinnerProposals()
	{
		return winners.toArray();
	}

	/**
	 *  Get the executed tasks.
	 *  @return The executed tasks.
	 */
	public Object[] getTasks()
	{
		return tasks.toArray();
	}

	/**
	 *  Get the winner participants.
	 *  @return The winner participants.
	 */
	public AgentIdentifier[] getWinnerParticipants()
	{
		List ret = new ArrayList();
		for(int i=0; i<winners.size(); i++)
		{
			ProposalInfo pi = (ProposalInfo)proposals.get(winners.get(i));
			ret.add(pi.getAgentIdentifier());
		}
		return (AgentIdentifier[])ret.toArray(new AgentIdentifier[ret.size()]);
	}

	/**
	 *  Get the non-acceptable proposals.
	 *  @return The non-acceptable proposals.
	 */
	public Object[] getNonAcceptableProposals()
	{
		HashSet ret = new HashSet(proposals.keySet());
		ret.removeAll(acceptables);
		return ret.toArray();
	}

	/**
	 *  Get the non-winner proposals.
	 *  @return The non-winner proposals.
	 */
	public Object[] getNonWinnerProposals()
	{
		HashSet ret = new HashSet(proposals.keySet());
		ret.removeAll(winners);
		return ret.toArray();
	}

	/**
	 *  Get the sender of a proposal.
	 *  @param proposal The proposal.
	 *  @return The sender of the proposal.
	 */
	public AgentIdentifier getAgentIdentifier(Object proposal)
	{
		ProposalInfo pi = (ProposalInfo)proposals.get(proposal);
		return pi.getAgentIdentifier();
	}

	/**
	 *  Get the message event for a proposal.
	 *  @param proposal The proposal.
	 *  @return The message event for a proposal.
	 */
	public IMessageEvent getMessageEvent(Object proposal)
	{
		ProposalInfo pi = (ProposalInfo)proposals.get(proposal);
		return pi.getMessageEvent();
	}

	/**
	 *  Get the proposal for the sender.
	 *  @param sender The sender.
	 *  @return The proposal.
	 */
	public Object getProposal(AgentIdentifier sender)
	{
		Object ret = null;
		Object[] ps = proposals.keySet().toArray();
		for(int i=0; ret==null && i<ps.length; i++)
		{
			ProposalInfo pi = (ProposalInfo)proposals.get(ps[i]);
			if(pi.getAgentIdentifier().equals(sender))
				ret = ps[i];
		}
		return ret;
	}

	/**
	 *  Get the call for proposal.
	 *  @return The call for proposal.
	 */
	public Object getCFP()
	{
		return cfp;
	}

	/**
	 *  Get the number of proposals.
	 *  @return The number of proposals.
	 */
	public int size()
	{
		return proposals.size();
	}

	/**
	 *  Get the negotiation state.
	 *  @return The state.
	 */
	public String getState()
	{
		return state;
	}

	/**
	 *  Set the negotiation state.
	 *  @param state The state to set.
	 */
	public void setState(String state)
	{
		this.state = state;
	}
	
	/**
	 *  Get the start time.
	 *  @return The starttime.
	 */
	public long getStarttime()
	{
		return starttime;
	}

	/**
	 *  Set the start time.
	 *  @param starttime The start time to set.
	 */
	public void setStarttime(long starttime)
	{
		this.starttime = starttime;
	}
	
	/**
	 *  Get the end time.
	 *  @return The endtime.
	 */
	public long getEndtime()
	{
		return endtime;
	}

	/**
	 *  Set the end time.
	 *  @param endtime The end time to set.
	 */
	public void setEndtime(long endtime)
	{
		this.endtime = endtime;
	}

	/** 
	 * Get the string representation.
	 * @return The string representation.
	 */
	public String toString()
	{
		SimpleDateFormat sdf = new SimpleDateFormat("dd.MM.yyyy.'-'HH:mm:ss ': '");
		StringBuffer ret = new StringBuffer();
		ret.append("NegotiationRecord(");
		ret.append("starttime: "+sdf.format(new Date(starttime))+", ");		
		ret.append("endtime: "+sdf.format(new Date(endtime))+", ");
		ret.append("cfp: "+cfp+", ");
		ret.append("proposals: "+proposals+", ");
		ret.append("winners: "+winners+", ");
		ret.append("state: "+state);
		ret.append(")");
		return ret.toString();
	}	
	
}

/**
 *  Struct for saving some proposal infos.
 */
class ProposalInfo
{
	/** The message with the proposal. */
	protected IMessageEvent msg;
	
	// Hack for making accessible the data to all callers
	// problem: wrong thread calling plan interface
	protected Object content;
	protected AgentIdentifier proposer;

	/**
	 *  Create a new proposal info.
	 *  @param msg The message.
	 */
	public ProposalInfo(IMessageEvent msg)
	{
		this.msg = msg;
		this.content = msg.getContent();
		this.proposer = (AgentIdentifier)msg.getParameter(SFipa.SENDER).getValue();
	}

	/**
	 *  Get the sender.
	 *  @return The sender.
	 */
	public AgentIdentifier getAgentIdentifier()
	{
		return proposer;
		//;return (AgentIdentifier)msg.getParameter(SFipa.SENDER).getValue();
	}

	/**
	 *  Get the message event.
	 *  @return The message event.
	 */
	public IMessageEvent getMessageEvent()
	{
		return msg;
	}

	/** 
	 *  Get the string representation.
	 *  @return The string representation.
	 */
	public String toString()
	{
		return "ProposalInfo(proposer: "+getAgentIdentifier()+", proposal: "+content+")"; 
	}
}
