package jadex.planlib;

import java.util.List;

import jadex.adapter.fipa.SFipa;
import jadex.runtime.GoalFailureException;
import jadex.runtime.IGoal;
import jadex.runtime.IMessageEvent;
import jadex.runtime.MessageEventFilter;
import jadex.runtime.Plan;
import jadex.runtime.TimeoutException;
import jadex.util.collection.SCollection;


/**
 *  This plan implements the receiver of the "FIPA Dutch Auction Interaction
 *  Protocol Specification" (XC00032 - Experimental).
 *  
 *  An English auction is one where bidders continously can increase the current
 *  offer until no one is willing to increase any more.
 */
public class EAReceiverPlan extends Plan
{
	/**
	 *  The plan body.
	 */
	public void body()
	{
		// Fetch the auction information.
		IMessageEvent me = (IMessageEvent)getInitialEvent();
		AuctionInfo auctioninfo = (AuctionInfo)me.getParameter(SFipa.CONTENT).getValue();
		getLogger().info(getAgentName()+": Received inform_start_auction message with AuctionInfo " +
			"start time: "+auctioninfo.getStarttime()+" Round time "+auctioninfo.getRoundTimeout()
			+" topic: "+auctioninfo.getTopic());
		if(auctioninfo.getRoundTimeout()<=0)
		{
			getLogger().warning(getAgentName()+"No round timeout specified");
			fail();
		}
		
		// Offer the possibility to decide not to participate in the auction
		boolean participate = true;
		try
		{
			IGoal dp = getScope().getGoalbase().createGoal("ea_decide_participation");
			dp.getParameter("auction_info").setValue(auctioninfo);
			dispatchSubgoalAndWait(dp);
			Boolean part = (Boolean)dp.getParameter("participate").getValue();
			participate = part==null? true: part.booleanValue();
		}
		catch(GoalFailureException e)
		{
			// Participate if no explicit decision was made.
			getLogger().info("Optional goal ea_decide_request has not been handled.");
		}

		// Wait for messages with the conversation-id of the initial event (inform_start_auction).
		String convid = (String)me.getParameter(SFipa.CONVERSATION_ID).getValue();
		MessageEventFilter mef = new MessageEventFilter(null);
		mef.addValue(SFipa.CONVERSATION_ID, convid);
		getWaitqueue().addFilter(mef);

		boolean running = true;
		Object winning_offer = null;
		//boolean won = false;
		long buftimeout = (long)(auctioninfo.getRoundTimeout()*1.1);
		List offers = SCollection.createArrayList();
		long firsttimeout = auctioninfo.getStarttime()==0 || (auctioninfo.getStarttime()-System.currentTimeMillis()<=0)
			? -1 : auctioninfo.getStarttime()-System.currentTimeMillis()+buftimeout;
		int missing_cnt = 0;
		boolean sitting_out = false; 
		//System.out.println(getAgentName()+" timeout: "+timeout);
			
		while(participate && running)
		{
			IMessageEvent cfp = null;
			try
			{
				// Calculates 
				getLogger().info(getAgentName()+" waiting for: "+firsttimeout+" "+buftimeout);
				cfp = (IMessageEvent)waitFor(mef, firsttimeout==-1? buftimeout: firsttimeout);
				getLogger().info(getAgentName()+" received cfp: "+cfp.getType());
				firsttimeout=-1;
				//lastcfptime = System.currentTimeMillis();
				
				if(cfp.getType().equals("ea_cfp"))
				{
					//System.out.println("new cfp: "+winning_offer);
					
					// If has sitting out last round and a new round has been
					// issued reset winning_offer and sitting_out and continue
					// the auction
					if(sitting_out)
					{
						sitting_out = false;
						winning_offer = null;
					}
					
					if(winning_offer==null)
					{
						// Instantiate make_proposal-goal with the offer of the received CFP.
						IGoal mp = createGoal("ea_make_proposal");
						Object offer = cfp.getContent();
						offers.add(offer);
						mp.getParameter("offer").setValue(offer);
						mp.getParameter("auction_info").setValue(auctioninfo);
						mp.getParameterSet("history").addValues(offers.toArray());
						try
						{
							dispatchSubgoalAndWait(mp, auctioninfo.getRoundTimeout());
						}
						catch(TimeoutException e)
						{
							getLogger().info(getAgentName() + e.getMessage());
						}
						
						Boolean leave = (Boolean)mp.getParameter("leave").getValue();
						Boolean accept = (Boolean)mp.getParameter("accept").getValue();
						if(leave!=null && leave.booleanValue())
						{
							getLogger().info(getAgentName() + " informs the initiator of the auction "
								+auctioninfo.getTopic()+" that it doesn't want to participate.");
						
							IMessageEvent leavemsg = getEventbase().createMessageEvent("ea_not_understood");
							leavemsg.getParameterSet(SFipa.RECEIVERS).addValue(
								me.getParameter(SFipa.SENDER).getValue());
							leavemsg.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(leavemsg);
							
							participate = false;
						}
						else if(accept!=null && accept.booleanValue())
						{
							//System.out.println(getAgentName()+" sending proposal: "+offer);
							// Send propsal.
							IMessageEvent acceptmsg = getEventbase().createMessageEvent("ea_propose");
							acceptmsg.getParameterSet(SFipa.RECEIVERS).addValue(
								me.getParameter(SFipa.SENDER).getValue());
							acceptmsg.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(acceptmsg);
						}
					}
					else
					{
						//System.out.println("Sitting out: "+winning_offer);
						sitting_out = true;
					}
				}
				else if(cfp.getType().equals("ea_accept_proposal"))
				{
					winning_offer = cfp.getContent();
				}
				else if(cfp.getType().equals("ea_reject_proposal"))
				{
					winning_offer = null;
				}
				else if(cfp.getType().equals("ea_inform_end_auction"))
				{
					// Check if really won the auction by comparing the
					// official winning offer and my last winning offer 
					// (could have missed a message)
					if(winning_offer!=null && !winning_offer.equals(cfp.getContent()))
						winning_offer = null;
					
					running = false;
				}
				else
				{
					getLogger().warning("Could not understand: "+cfp+" "+cfp.getType());
				}
			}
			catch(TimeoutException e)
			{
				getLogger().info(getAgentName()+" "+e.getMessage());
				// Exit when no offers are received any more (for 3 times).
				//System.out.println(getAgentName()+" missed cfp: "+missing_cnt);
				if(++missing_cnt==3)
					running = false; 
			}
		}
		
		// Do not create the finished info goal when left the auction
		if(!running)
		{
			IGoal af = createGoal("ea_auction_finished");
			af.getParameter("offer").setValue(winning_offer);
			af.getParameter("auction_info").setValue(auctioninfo);
			af.getParameter("won").setValue(new Boolean(winning_offer!=null));
			af.getParameterSet("history").addValues(offers.toArray());
			try
			{
				dispatchSubgoalAndWait(af);
			}
			catch(GoalFailureException e)
			{
				getLogger().info(getAgentName() + ": No reaction on finished auction: "+offers);
			}
		}
		
		getWaitqueue().removeFilter(mef);
	}
}
