package jadex.planlib;

import java.util.List;

import jadex.adapter.fipa.AgentIdentifier;
import jadex.adapter.fipa.SFipa;
import jadex.runtime.IMessageEvent;
import jadex.runtime.MessageEventFilter;
import jadex.runtime.Plan;
import jadex.runtime.TimeoutException;
import jadex.util.SUtil;


/**
 *  This plan implements the initiator of the "FIPA English Auction Interaction
 *  Protocol Specification" (XC00031 - Experimental).
 *  
 *  An English auction is one where bidders continously can increase the current
 *  offer until no one is willing to increase any more.
 */
public class EAInitiatorPlan extends Plan
{
	/**
	 *  The plan body.
	 */
	public void body()
	{
		AuctionInfo auctioninfo = (AuctionInfo)getParameter("auction_info").getValue();
		if(auctioninfo.getRoundTimeout()<=0)
		{
			getLogger().warning(getAgentName()+"No round timeout specified");
			fail();
		}
		
		long timetowait = auctioninfo.getStarttime()==0? 0: 
			auctioninfo.getStarttime() - System.currentTimeMillis();
		// Fetch the timeout for each round of the auction.
		long roundtimeout = auctioninfo.getRoundTimeout();
		
		// The offer calculator is responsible for calculating the value of the
		// current round and also contains the start and the minimal offers.
		IOfferGenerator offercalc = (IOfferGenerator)getParameter("offer_generator").getValue();

		// Fetch the receivers.
		List receivers = SUtil.arrayToList(getParameterSet("receivers").getValues());

		// Send the inform_start_auction-message to all receivers.
		IMessageEvent start = getEventbase().createMessageEvent("ea_inform_start_auction");
		start.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
		String convid = (String)start.getParameter(SFipa.CONVERSATION_ID).getValue();
		start.getParameter(SFipa.CONTENT).setValue(auctioninfo);
		getLogger().info(getAgentName() + ":\tinform_start_auction");
		
		MessageEventFilter mef = new MessageEventFilter(null);
		mef.addValue(SFipa.CONVERSATION_ID, convid);
		getWaitqueue().addFilter(mef);
		
		sendMessage(start);
		
		// The initiator of the interaction protocol shall wait until interested
		// agents are ready to participate.
		// If agents indicate that they do not wish to participate they are excluded
		// from the auction.
		//System.out.println(getAgentName()+" waiting for: "+timetowait);
		while(timetowait > 0)
		{
			IMessageEvent removebidder;
			try
			{
				removebidder = (IMessageEvent)waitFor(mef, timetowait);
			}
			catch(TimeoutException e)
			{
				break;
			}
			
			if(removebidder.getType().equals("ea_not_understood"))
			{
				receivers.remove(removebidder.getParameter(SFipa.SENDER).getValue());
				getLogger().info("Removed "+((AgentIdentifier)removebidder.getParameter(SFipa.SENDER).getValue()).getName() + ".");
			}
			else
			{
				getLogger().warning("Could not handle message of type "+removebidder.getType() 
					+" from "+((AgentIdentifier)removebidder.getParameter(SFipa.SENDER).getValue()).getName()+".");
			}
			
			timetowait =  - System.currentTimeMillis();
		}
		
		// Send calls for proposal until no more proposals are received.
		boolean running = true;
		Comparable winning_offer = null;
		IMessageEvent winner = null;
		
		while(running && receivers.size()>0 
			&& offercalc.getCurrentOffer().compareTo(offercalc.getLimitOffer())<0)
		{
			Comparable offer = offercalc.getCurrentOffer();
			//System.out.println(getAgentName()+" current offer is "+offer);
				
			// Send CFP.
			IMessageEvent cfp = getEventbase().createMessageEvent("ea_cfp");
			cfp.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
			cfp.setContent(offer);
			cfp.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
			getLogger().info(getAgentName() + ": cfp(" + offer + ")");
			sendMessage(cfp);

			IMessageEvent first_proposal = null;

			//The first proposal is accepted, so wait until first the proposal is received.
			long roundstart = System.currentTimeMillis();
			while(System.currentTimeMillis() - roundstart < roundtimeout)
			{
				try
				{
					//System.out.println(getAgentName()+" waiting for accepts for: "+offer);
					IMessageEvent tmp = (IMessageEvent)waitFor(mef, roundtimeout);
					
					if(tmp.getType().equals("ea_propose"))
					{
						if(first_proposal==null)
						{
							getLogger().info(getAgentName()+" got first accept for: "
								+offer+" from: "+tmp.getParameter(SFipa.SENDER).getValue());
							// Send the accept_proposal-message to the agent with the first proposal.
							IMessageEvent accept = tmp.createReply("ea_accept_proposal");
							accept.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							accept.getParameter(SFipa.CONTENT).setValue(offer);
							sendMessage(accept);
							first_proposal = tmp;
						}
						else
						{
							getLogger().info(getAgentName()+" got too late accept for: "
								+offer+" from: "+tmp.getParameter(SFipa.SENDER).getValue());
							// Send reject_proposal-message.
							IMessageEvent reject = tmp.createReply("ea_reject_proposal");
							reject.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(reject);
						}
					}
					else
					{
						// Remove agent from the list of receivers on any other of
						// message. So you can use e.g. a not_understood_message to exit the auction.
						getLogger().info(getAgentName()+" removing agent "
							+tmp.getParameter(SFipa.SENDER).getValue());
						receivers.remove(tmp.getParameter(SFipa.SENDER).getValue());
					}
				}
				catch(TimeoutException e)
				{
					getLogger().info("No further bids in this round");
				}
			}

			// Set the winner if propsals have been received, otherwise
			// cease sending CFP-messages (so the winner of the last round will
			// be the winner of the auction).
			if(first_proposal != null)
			{
				winner = first_proposal;
				winning_offer = offer;
				offercalc.setNextRound();
			}
			else
			{
				// End when no proposals have been received.
				running = false;
			}
		}
		
		if(winner!=null)
		{
			AgentIdentifier wina = (AgentIdentifier)winner.getParameter(SFipa.SENDER).getValue();
			getLogger().info(getAgentName() + ": AUCTION TERMINATED (winner: "
				+wina.getName()+" - price: "+winning_offer+")");
			getParameter("winner").setValue(wina);
			getParameter("winning_offer").setValue(winning_offer);
		}
		else
		{ 	
			getLogger().info(getAgentName()+ ": AUCTION TERMINATED "+
				"(no winner - initiator didn't receive any proposals)");
		}
		
		// Send the inform_end_auction-message.
		IMessageEvent end = getEventbase().createMessageEvent("ea_inform_end_auction");
		end.getParameter(SFipa.CONTENT).setValue(winning_offer);
		end.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
		end.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
		sendMessage(end);
		
		getWaitqueue().removeFilter(mef);
	}
}
