package jadex.planlib;

import java.util.List;

import jadex.adapter.fipa.SFipa;
import jadex.runtime.GoalFailureException;
import jadex.runtime.IGoal;
import jadex.runtime.IMessageEvent;
import jadex.runtime.MessageEventFilter;
import jadex.runtime.Plan;
import jadex.runtime.TimeoutException;
import jadex.util.collection.SCollection;


/**
 *  This plan implements the receiver of the "FIPA Dutch Auction Interaction
 *  Protocol Specification" (XC00032 - Experimental).
 *  
 *  A dutch auction is one where the auctioneer starts with a high start price
 *  and continually lowers it until the first bidder accepts the price.
 */
public class DAReceiverPlan extends Plan
{
	/**
	 * The plan body.
	 */
	public void body()
	{
		// Fetch the auction information.
		IMessageEvent me = (IMessageEvent)getInitialEvent();
		AuctionInfo auctioninfo = (AuctionInfo)me.getParameter(SFipa.CONTENT).getValue();
		getLogger().info(getAgentName()+": Received inform_start_auction message with AuctionInfo " +
			"start time: "+auctioninfo.getStarttime()+" Round time "+auctioninfo.getRoundTimeout()
			+" topic: "+auctioninfo.getTopic());
		if(auctioninfo.getRoundTimeout()<=0)
		{
			getLogger().warning(getAgentName()+"No round timeout specified");
			fail();
		}
		
		// Offer the possibility to decide not to participate in the auction
		boolean participate = true;
		try
		{
			IGoal dp = getScope().getGoalbase().createGoal("da_decide_participation");
			dp.getParameter("auction_info").setValue(auctioninfo);
			dispatchSubgoalAndWait(dp);
			Boolean part = (Boolean)dp.getParameter("participate").getValue();
			participate = part==null? true: part.booleanValue();
		}
		catch(GoalFailureException e)
		{
			// Participate if no explicit decision was made.
			getLogger().info("Optional goal da_decide_request has not been handled.");
		}

		// Wait for messages with the conversation-id of the initial event (inform_start_auction).
		String convid = (String)me.getParameter(SFipa.CONVERSATION_ID).getValue();
		MessageEventFilter mef = new MessageEventFilter(null);
		mef.addValue(SFipa.CONVERSATION_ID, convid);
		getWaitqueue().addFilter(mef);

		boolean running = true;
		boolean won = false;
		long buftimeout = (long)(auctioninfo.getRoundTimeout()*1.1);
		List offers = SCollection.createArrayList();
		long firsttimeout = auctioninfo.getStarttime()==0 || (auctioninfo.getStarttime()-System.currentTimeMillis()<=0)
			? -1 : auctioninfo.getStarttime()-System.currentTimeMillis()+buftimeout;
		//long lastcfptime = System.currentTimeMillis();
		int missing_cnt = 0;
		//System.out.println(getAgentName()+" timeout: "+timeout);
			
		while(participate && running)
		{
			IMessageEvent cfp = null;
			try
			{
				// Calculates 
				getLogger().info(getAgentName()+" waiting for: "+firsttimeout+" "+buftimeout);
				cfp = (IMessageEvent)waitFor(mef, firsttimeout==-1? buftimeout: firsttimeout);
				getLogger().info(getAgentName()+" received cfp: "+cfp.getContent());
				firsttimeout=-1;
				
				if(cfp.getType().equals("da_cfp"))
				{
					// Instantiate make_proposal-goal with the offer of the received CFP.
					IGoal mp = createGoal("da_make_proposal");
					Object offer = cfp.getContent();
					offers.add(offer);
					mp.getParameter("offer").setValue(offer);
					mp.getParameter("auction_info").setValue(auctioninfo);
					mp.getParameterSet("history").addValues(offers.toArray());
					Boolean leave = null;
					try
					{
						dispatchSubgoalAndWait(mp, auctioninfo.getRoundTimeout());
						leave = (Boolean)mp.getParameter("leave").getValue();
					}
					catch(TimeoutException e)
					{
						getLogger().info(getAgentName() + e.getMessage());
					}
					
					if(leave!=null && leave.booleanValue())
					{
						getLogger().info(getAgentName() + " informs the initiator of the auction "
							+auctioninfo.getTopic()+" that it doesn't want to participate.");
					
						IMessageEvent leavemsg = getEventbase().createMessageEvent("da_not_understood");
						leavemsg.getParameterSet(SFipa.RECEIVERS).addValue(
							me.getParameter(SFipa.SENDER).getValue());
						leavemsg.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
						sendMessage(leavemsg);
						
						participate = false;
					}
					else if(((Boolean)mp.getParameter("accept").getValue()).booleanValue())
					{
						getLogger().info(getAgentName()+" sending proposal: "+offer);
						// Send propsal.
						IMessageEvent accept = getEventbase().createMessageEvent("da_propose");
						accept.getParameterSet(SFipa.RECEIVERS).addValue(
							me.getParameter(SFipa.SENDER).getValue());
						accept.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
						sendMessage(accept);
					}
				}
				else if(cfp.getType().equals("da_accept_proposal"))
				{
					won = true;
					running = false;
				}
				else if(cfp.getType().equals("da_reject_proposal") 
					|| cfp.getType().equals("da_inform_end_auction"))
				{
					running = false;
				}
				else
				{
					System.out.println("Could not understand: "+cfp+" "+cfp.getType());
				}
			}
			catch(TimeoutException e)
			{
				getLogger().info(getAgentName()+" "+e.getMessage());
				// Exit when no offers are received any more (for 3 times).
				//System.out.println(getAgentName()+" missed cfp: "+missing_cnt);
				if(++missing_cnt==3)
					running = false; 
			}
		}
		
		// Do not create the finished info goal when left the auction
		if(!running)
		{
			IGoal af = createGoal("da_auction_finished");
			if(offers.size()>0)
				af.getParameter("offer").setValue(offers.get(offers.size()-1));
			af.getParameter("auction_info").setValue(auctioninfo);
			af.getParameter("won").setValue(new Boolean(won));
			af.getParameterSet("history").addValues(offers.toArray());
			try
			{
				dispatchSubgoalAndWait(af);
			}
			catch(GoalFailureException e)
			{
				getLogger().info(getAgentName() + ": No reaction on finished auction: "+offers);
			}
		}
		
		getWaitqueue().removeFilter(mef);
	}
}
