package jadex.planlib;

import java.util.List;

import jadex.adapter.fipa.AgentIdentifier;
import jadex.adapter.fipa.SFipa;
import jadex.runtime.IMessageEvent;
import jadex.runtime.MessageEventFilter;
import jadex.runtime.Plan;
import jadex.runtime.TimeoutException;
import jadex.util.SUtil;


/**
 *  This plan implements the initiator of the "FIPA Dutch Auction Interaction
 *  Protocol Specification" (XC00032 - Experimental)
 *  
 *  A dutch auction is one where the auctioneer starts with a high start price
 *  and continually lowers it until the first bidder accepts the price.
 */
public class DAInitiatorPlan extends Plan
{
	/**
	 * The plan body.
	 */
	public void body()
	{
		AgentIdentifier winner = null;
		AuctionInfo auctioninfo = (AuctionInfo)getParameter("auction_info").getValue();
		if(auctioninfo.getRoundTimeout()<=0)
		{
			getLogger().warning(getAgentName()+"No round timeout specified");
			fail();
		}
		
		long timetowait = auctioninfo.getStarttime()==0? 0: 
			auctioninfo.getStarttime() - System.currentTimeMillis();
		// Fetch the timeout for each round of the auction.
		long roundtimeout = auctioninfo.getRoundTimeout();
		
		// The offer calculator is responsible for calculating the value of the
		// current round and also contains the start and the minimal offers.
		IOfferGenerator offercalc = (IOfferGenerator)getParameter("offer_generator").getValue();

		// Fetch the receivers.
		List receivers = SUtil.arrayToList(getParameterSet("receivers").getValues());

		// Send the inform_start_auction-message to all receivers.
		IMessageEvent start = getEventbase().createMessageEvent("da_inform_start_auction");
		start.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
		String convid = (String)start.getParameter(SFipa.CONVERSATION_ID).getValue();
		start.getParameter(SFipa.CONTENT).setValue(auctioninfo);
		getLogger().info(getAgentName() + ":\tinform_start_auction");
		
		MessageEventFilter mef = new MessageEventFilter(null);
		mef.addValue(SFipa.CONVERSATION_ID, convid);
		getWaitqueue().addFilter(mef);
		
		sendMessage(start);

		// The initiator of the interaction protocol shall wait until interested
		// agents are ready to participate.
		// If agents indicate that they do not wish to participate they are excluded
		// from the auction.
		//System.out.println(getAgentName()+" waiting for: "+timetowait);
		while(timetowait > 0)
		{
			IMessageEvent removebidder;
			try
			{
				removebidder = (IMessageEvent)waitFor(mef, timetowait);
			}
			catch(TimeoutException e)
			{
				break;
			}
			
			if(removebidder.getType().equals("da_not_understood"))
			{
				receivers.remove(removebidder.getParameter(SFipa.SENDER).getValue());
				getLogger().info("Removed "+((AgentIdentifier)removebidder.getParameter(SFipa.SENDER).getValue()).getName() + ".");
			}
			else
			{
				getLogger().warning("Could not handle message of type "+removebidder.getType() 
					+" from "+((AgentIdentifier)removebidder.getParameter(SFipa.SENDER).getValue()).getName()+".");
			}
			
			timetowait =  - System.currentTimeMillis();
		}

		// Send calls for proposal until the limit price is reached or an agent is
		// willing to pay the actual price.
		// Auction ends when winner is determined, limit price is reached or
		// no receiver is left.
		while(winner==null && receivers.size()>0 
			&& offercalc.getCurrentOffer().compareTo(offercalc.getLimitOffer())>0)
			// && offercalc.getCurrentOffer() > offercalc.getLimitOffer())
		{
			// Send CFP to all participants.
			IMessageEvent cfp = getEventbase().createMessageEvent("da_cfp");
			cfp.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
			cfp.setContent(offercalc.getCurrentOffer());
			cfp.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
			getLogger().info(getAgentName() + ": cfp(price: " + offercalc.getCurrentOffer() + ") "+receivers);
			
			sendMessage(cfp);
			
			// Perform a negotiation round as long as no winner could be determined.
			long roundstart = System.currentTimeMillis();
			while(System.currentTimeMillis() - roundstart < roundtimeout)
			{
				IMessageEvent reply = null;
				try
				{
					reply = (IMessageEvent)waitFor(mef, roundtimeout);
					if(reply.getType().equals("da_propose"))
					{
						// Accept the first winner
						if(winner==null)
						{
							// Send the accept_proposal-message to the agent with the first proposal.
							IMessageEvent accept = getEventbase().createMessageEvent("da_accept_proposal");
							accept.getParameterSet(SFipa.RECEIVERS).addValue(reply.getParameter(SFipa.SENDER).getValue());
							accept.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(accept);
							getLogger().info(getAgentName() + " found winner: "+reply.getParameter(SFipa.SENDER).getValue());
							
							// Send the inform_end_auction-message.
							IMessageEvent end = getEventbase().createMessageEvent("da_inform_end_auction");
							end.getParameterSet(SFipa.RECEIVERS).addValues(receivers.toArray());
							end.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(end);
							getLogger().info(getAgentName() + ": inform_end_auction");
							
							// Set the parameter "winner" to the AgentIdentifier of the
							// winning agent.
							winner = (AgentIdentifier)reply.getParameter(SFipa.SENDER).getValue();
							getParameter("winner").setValue(winner);
							getParameter("winning_offer").setValue(offercalc.getCurrentOffer());
						}
						// Reject all other proposals
						else
						{
							// Send reject_proposal-message.
							IMessageEvent tmp = reply.createReply("da_reject_proposal");
							tmp.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
							sendMessage(tmp);
						}
					}
					else
					{
						// Remove agent from the list of receivers on any
						// other of message. So you can use e.g. a
						// not_understood_message to exit the auction
						receivers.remove(reply.getParameter(SFipa.SENDER).getValue());
					}
				}
				catch(TimeoutException e)
				{
				}
			}
			
			if(winner==null)
				offercalc.setNextRound();
		}
			
		if(winner == null)
		{
			getLogger().info(getAgentName() + ": AUCTION TERMINATED (no winner)");
		}
		else
		{
			getLogger().info(getAgentName() + ": AUCTION TERMINATED (winner: " 
				+winner.getName() + " - price: " + offercalc.getCurrentOffer() + ")");
		}
		
		getWaitqueue().removeFilter(mef);
	}
}

