package jadex.planlib;

import jadex.runtime.*;
import jadex.adapter.fipa.SFipa;
import jadex.adapter.fipa.AgentIdentifier;
import jadex.util.SUtil;

import java.util.*;

/**
 *  Handles the initiator side of a cfp protocol.
 */
public class CNPInitiatorPlan extends Plan
{
	//-------- attributes --------

	/** The message filter for all protocol related messages. */
	protected MessageEventFilter mf;

	/** The timeout. */
	protected long timeout;

	/** The selector. */
	protected ISelector selector;

	/** Arbitrary send message with convid for receiving answer messages. */
	// todo: hack, must save at least one message for being able to wait for replies. Otherwise gc would cleanup all messages.
	protected IMessageEvent me;

	//-------- constructors --------

	/**
	 *  The body method is called on the
	 *  instatiated plan instance from the scheduler.
	 */
	public void body()
	{
		// Initialize negotiations.

		if(hasParameter("timeout") && getParameter("timeout").getValue()!=null)
		{
			timeout = ((Long)getParameter("timeout").getValue()).longValue();
		}
		else if(getBeliefbase().containsBelief("timeout") && getBeliefbase().getBelief("timeout").getFact()!=null)
		{
			timeout = ((Long)getBeliefbase().getBelief("timeout").getFact()).longValue();
		}
		else
		{
			timeout = -1;
		}

		String convid = SFipa.createUniqueId(getAgentName());
		mf = new MessageEventFilter(null);
		mf.addValue(SFipa.CONVERSATION_ID, convid);
		getWaitqueue().addFilter(mf);
		selector = (ISelector)getParameter("selector").getValue();
		IIterationDecider decider = (IIterationDecider)getParameter("iteration_decider").getValue();

		// Start negotiations.

		AgentIdentifier[] participants = (AgentIdentifier[])getParameterSet("receivers").getValues();
		NegotiationRecord nr = new NegotiationRecord(getParameter("content").getValue());
		nr.setStarttime(System.currentTimeMillis());
		getParameterSet("history").addValue(nr);

		// Perform negotiation rounds.

		while(participants.length>0)
		{
			// Send proposals.
			sendProposals(participants, nr.getCFP(), convid);

			// Collect proposals.
			collectProposals(nr, participants);

			// Determine winners.
			determineWinners(nr);

			if(decider==null)
				break;

			// Compute next round participants.
			NegotiationRecord[] nrs = (NegotiationRecord[])getParameterSet("history").getValues();
			AgentIdentifier[] newparticipants = decider.decideIteration(nrs);

			if(newparticipants.length>0)
			{
				// Immediately reject excluded participants.
				rejectExcludedProposals(nr, participants, newparticipants);
				nr.setEndtime(System.currentTimeMillis());
				
				// Compute next round refined cfp.
				Object cfp = decider.refineCFP(nrs);
				nr = new NegotiationRecord(cfp);
				nr.setStarttime(System.currentTimeMillis());
				getParameterSet("history").addValue(nr);
			}
			participants = newparticipants;
		}

		// Finish protocols by contracting winner proposal senders.

		// Answers proposals.
		answerProposals(nr);

		// Wait for tasks done.
		waitForTasksDone(nr);
		nr.setEndtime(System.currentTimeMillis());
		
		// Determine failure.
		determineFailure(nr);//, convid);

		getParameterSet("result").addValues(nr.getTasks());
		getLogger().info(getAgentName()+"CNPPlan finished: "+convid);
	}

	/**
	 *  Send the proposal message.
	 *  @param participants The participants.
	 *  @param cfp The call for proposal.
	 *  @param convid The conversation id.
	 */
	protected void sendProposals(AgentIdentifier[] participants, Object cfp, String convid)
	{
		//IMessageEvent me = getEventbase().createMessageEvent("cnp_cfp");
		me = getEventbase().createMessageEvent("cnp_cfp");
		me.getParameterSet(SFipa.RECEIVERS).addValues(participants);
		me.getParameter(SFipa.CONVERSATION_ID).setValue(convid);
		me.setContent(cfp);
		getLogger().info(getAgentName()+" CNPPlan initiated: "+convid);
		sendMessage(me);
	}

	/**
	 *  Send the proposal message.
	 *  @param pr The negotiation record.
	 *  @param participants The participants.
	 */
	protected void collectProposals(NegotiationRecord pr, AgentIdentifier[] participants)
	{
		List rec = SUtil.arrayToList(participants);

		long time = System.currentTimeMillis();
		try
		{
			while(rec.size() > 0)
			{
				// Wait for the replies.
				long wait_time = timeout + time - System.currentTimeMillis();
				if(wait_time <= 0)
					break;

				getLogger().info(getAgentName()+" CNPPlan: waiting: "+wait_time);

				//IMessageEvent reply = waitForReply(me, wait_time);
				IMessageEvent reply = (IMessageEvent)waitFor(mf, wait_time);
				AgentIdentifier sender = (AgentIdentifier)reply.getParameter(SFipa.SENDER).getValue();
				rec.remove(sender);
				if(reply.getType().equals("cnp_propose"))
				{
					getLogger().info(getAgentName()+" CNPPlan received a proposal reply: "+reply.getContent());
					//proposals.put(reply.getContent(), reply);
					pr.addProposal(reply);
				}
			}
		}
		catch(TimeoutException e)
		{
			// nop
		}
	}

	/**
	 *  Determine winners.
	 *  @param nr The negotiation record.
	 */
	protected void determineWinners(NegotiationRecord nr)
	{
		// Determine winner(s).
		Object[] tmp = selector.select(nr.getProposals());
		nr.setAcceptableProposals((Object[])tmp[0]); // the acceptables
		nr.setWinnerProposals((Object[])tmp[1]); // the winners
		getLogger().info(getAgentName()+" CNPPlan determined acceptables: "
			+SUtil.arrayToString(tmp[0])+" winners: "+SUtil.arrayToString(tmp[1]));
	}

	/**
	 *  Reject proposals.
	 *  @param nr negotiation record.
	 *  @param participants The participants.
	 *  @param newparticipants The new participants.
	 */
	protected void rejectExcludedProposals(NegotiationRecord nr, Object[] participants, Object[] newparticipants)
	{
		Set excluded = SUtil.arrayToSet(participants);
		excluded.removeAll(SUtil.arrayToSet(newparticipants));
		Object[] excluded_proposals = excluded.toArray();
		for(int i=0; i<excluded_proposals.length; i++)
		{
			IMessageEvent rej = nr.getMessageEvent(excluded_proposals[i]).createReply("cnp_reject");
			sendMessage(rej);
		}
	}

	/**
	 *  Accept and reject proposals.
	 *  @param nr negotiation record.
	 */
	protected void answerProposals(NegotiationRecord nr)
	{
		// Send accept messages to winners.
		// Map sender -> proposals

		Object[] win_proposals = nr.getWinnerProposals();
		for(int i=0; i<win_proposals.length; i++)
		{
			IMessageEvent acc = nr.getMessageEvent(win_proposals[i]).createReply("cnp_accept");
			sendMessage(acc);
		}

		// Send reject messages.
		Object[] other_proposals = nr.getNonWinnerProposals();
		for(int i=0; i<other_proposals.length; i++)
		{
			IMessageEvent rej = nr.getMessageEvent(other_proposals[i]).createReply("cnp_reject");
			sendMessage(rej);
		}
	}

	/**
	 *  Wait for tasks done.
	 *  @param nr The negotiation record.
	 */
	protected void waitForTasksDone(NegotiationRecord nr)
	{
		List participants = SUtil.arrayToList(nr.getWinnerParticipants());

		long time = System.currentTimeMillis();
		try
		{
			while(participants.size() > 0)
			{
				// Wait for the replies.
				long wait_time = timeout + time - System.currentTimeMillis();
				if(wait_time <= 0)
					break;

				IMessageEvent reply = (IMessageEvent)waitFor(mf, wait_time);
				AgentIdentifier sender = (AgentIdentifier)reply.getParameter(SFipa.SENDER).getValue();
				participants.remove(sender);

				if(reply.getType().equals("cnp_inform_done"))
				{
					// can be inform done
					nr.addExecutedTask(nr.getProposal(sender));
				}
				/*else if(reply.getType().equals("cnp_inform_result"))
				{
					// or inform result
				}*/
				else
				{
					getLogger().info("One task was possibly not executed: "+this+" "+getAgentName()+" "+sender);
				}
			}
		}
		catch(TimeoutException e)
		{
			// nop
		}
	}

	/**
	 *  Determine failure.
	 *  @param nr The negotiation record.
	 */
	protected void determineFailure(NegotiationRecord nr)//, String convid)
	{
		boolean needall = ((Boolean)getParameter("needall").getValue()).booleanValue();

		int done_cnt = nr.getTasks().length;
		if((!needall && done_cnt==0) ||  (needall && done_cnt!=nr.getWinnerProposals().length))
		{
			getLogger().info(getAgentName()+" CNPPlan failed: ");//+convid);
			nr.setState(NegotiationRecord.FAILED);
			fail();
		}
		nr.setState(NegotiationRecord.SUCCEEDED);
	}
}
